/*
  (c) 2003 Martijn van Oosterhout, kleptog@svana.org
  (c) 2002 Bertrik Sikken, bertrik@zonnet.nl
  
  HP5400/5470 Test util.
  Currently is only able to read back the scanner version string,
  but this basically demonstrates ability to communicate with the scanner.
  
  Massively expanded. Can do calibration scan, upload gamma and calibration
  tables and stores the results of a scan. - 19/02/2003 Martijn
  
*/

#include <stdio.h>      /* for printf */
#include <stdlib.h>     /* for exit */
#include <assert.h>
#include <stdint.h>
#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <math.h>
#include <netinet/in.h>  /* for htons */

#include "hp5400_xfer.h"


#define CMD_GETVERSION   0x1200
//#define CMD_SETUITEXT   
#define CMD_GETUITEXT    0xf00b
#define CMD_GETCMDID     0xc500
#define CMD_SCANREQUEST  0x2500
#define CMD_SCANRESPONSE 0x3400

/* Testing stuff to make it work */
#define CMD_SETDPI       0x1500   /* ??? */
#define CMD_USEGAMMA     0x1B01   /* 0x40 = no, 0x00 = yes */
#define CMD_UNKNOWN      0x2300   /* Send fixed string */
#define CMD_UNKNOWN2     0xD600   /* ??? Set to 0x04 */
#define CMD_UNKNOWN3     0xC000   /* ??? Set to 02 03 03 3C */

const char MatchVersion[] = "SilitekIBlizd C3 ScannerV0.84";

int WriteByte( int iHandle, int cmd, char data )
{
  if (hp5400_command_write(iHandle, cmd, 1, &data) < 0) {
    printf("failed to send byte (cmd=%04X)\n", cmd);
    return -1;
  }
  return 0;
}

int WarmupLamp( int iHandle )
{
  int i = 30;  // Max 30 seconds, 15 is typical for cold start?
  
  // Keep writing 01 to 0000 until no error...
  while( i > 0 )
  {
    if( WriteByte( iHandle, 0x0000, 0x01 ) == 0 )
      return 0;
    sleep(1);
    i--;
  }

  printf("***WARNING*** Warmup lamp failed...\n");
  return -1;
}

int SetGamma( int iHandle )
{
  short *buffer;
  char cmd[8];
  int i;
  struct CalPixel { 
    short highr, highg, highb;
    short lowr, lowg, lowb;
  };
  struct CalBlock {
    struct CalPixel pixels[42];
    char pad[8];
  } __attribute__((packed));
  
  struct CalBlock *calinfo;
  
  // We need 2690 CalPixels. A CalBlock is 512 bytes and has 42 CalPixels.
  // 2690/42 = 64 rem 2, so we need 65 blocks
  calinfo = malloc( 64 * 512 + 24 );  // 0x8018

  bzero( calinfo, 0x8018 );  
  for( i=0; i<2690; i++ )
  {
    struct CalPixel *pixel = & calinfo[ i / 42 ].pixels[ i % 42 ];

    // For now they're hardcoded. These approximate the values sent in windows.    
    pixel->highr = 0x6700;
    pixel->highg = 0x5e00;
    pixel->highb = 0x6000;
  
    pixel->lowr = 0x0530;
    pixel->lowg = 0x0530;
    pixel->lowb = 0x0530;
  }
  
  cmd[0] = 0x00; cmd[1] = 0x80; cmd[2] = 0x18; cmd[3] = 0x00;
  cmd[4] = 0x54; cmd[5] = 0x02; cmd[6] = 0x80; cmd[7] = 0x00;
  
  hp5400_bulk_command_write( iHandle, 0xE603, cmd, 8, 0x8018, 0x8018, (void*)calinfo );
  
  free( calinfo );
  
  // Setup dummay gamma correction table
  buffer = malloc( 2 * 65536 );
  for( i=0; i<65536; i++ )
    buffer[i] = i;
    
  cmd[0] = 2;
  cmd[1] = 0;
  cmd[2] = 0;
    
  hp5400_bulk_command_write( iHandle, 0x2A01, cmd, 3, 2 * 65536, 65536, (void*)buffer );
  hp5400_bulk_command_write( iHandle, 0x2A02, cmd, 3, 2 * 65536, 65536, (void*)buffer );
  hp5400_bulk_command_write( iHandle, 0x2A03, cmd, 3, 2 * 65536, 65536, (void*)buffer );
  
  free( buffer );

  return 0;
}

// bpp is BYTES per pixel
void DecodeImage( FILE *file, int planes, int bpp, int xsize, int ysize, char *filename )
{
  char *in, *buf;
  char *p[3];
  FILE *output;
  int i,j,k;

  // xsize is byte width, not pixel width
  xsize /= planes*bpp;
  
  printf("DecodeImage(planes=%d,bpp=%d,xsize=%d,ysize=%d) => %d\n", planes, bpp, xsize, ysize, planes*bpp*xsize*ysize );
  in = malloc( planes * (xsize*bpp + 1) );
  
  for( i=0; i<planes; i++ )
    p[i] = in + i*(xsize*bpp + 1);
    
  buf = malloc( 3*xsize*bpp );
  
  output = fopen( filename, "wb" );
  
  fprintf( output, "P%d\n%d %d\n", (planes==3)?6:5, xsize, ysize );
  fprintf( output, "%d\n", (bpp==1)?255:0xb000 );
  
  for( i=0; i<ysize; i++ )
  {
    fread( in, planes * (xsize*bpp + 1), 1, file );
    
    for( j=0; j<xsize; j++ )
    {
      for( k=0; k<planes; k++ )
      {
        if( bpp == 1 )
        {
          buf[j*planes + k] = p[k][j];
        }
        else if( bpp == 2 )
        {
          // Scanner sends MSB, PBM need MSB
          buf[j*planes*2 + k*2 + 0] = p[k][2*j+0];
          buf[j*planes*2 + k*2 + 1] = p[k][2*j+1];
        }
      }
    }
    fwrite( buf, planes*xsize*bpp, 1, output );
  }
  
  fclose( output );
  
  free( in );
  free( buf );
}

struct ScanRequest
{
  uint8_t x1;                       /* Set to 0x08 */
  uint16_t dpix, dpiy;              /* Set to 75, 150 or 300 in network order */
  uint16_t offx, offy;              /* Offset to scan, in 1/300th of dpi, in network order */
  uint16_t lenx, leny;              /* Size of scan, in 1/300th of dpi, in network order */
  uint16_t flags1, flags2, flags3;  /* Undetermined flag info */
     /* Known combinations are:
         1st calibration scan: 0x0000, 0x0010, 0x1820  =  24bpp
         2nd calibration scan: 0x0000, 0x0010, 0x3020  =  48bpp ???
         3rd calibration scan: 0x0000, 0x0010, 0x3024  =  48bpp ???
         Preview scan:         0x0080, 0x0000, 0x18E8  =   8bpp
         4th & 5th like 2nd and 3rd
         B&W scan:             0x0080, 0x0040, 0x08E8  =   8bpp
         6th & 7th like 2nd and 3rd
         True colour scan      0x0080, 0x0040, 0x18E8  =  24bpp
     */
  uint8_t zero;                     /* Seems to always be zero */
  uint16_t gamma[3];                /* Set to 100 in network order. Gamma? */
  uint16_t pad[3];                  /* Zero padding ot 32 bytes??? */               
} __attribute__((packed));

struct ScanResponse
{
  uint16_t x1;                        /* Usually 0x0000 or 0x4000 */
  uint32_t transfersize;              /* Number of bytes to be transferred */
  uint16_t zero;                      /* Seems to be zero */
  uint16_t xsize, ysize;              /* Shape of returned bitmap */
  uint16_t pad[2];                      /* Zero padding to 16 bytes??? */
} __attribute__((packed));

int DoScan( int iHandle, struct ScanRequest *req, char *filename, int code )
{
  FILE *file;
  struct ScanResponse res;
    
  file = fopen( filename, "w+b" );
  if( !file )
  {
    printf( "Couldn't open outputfile (%s)\n", strerror(errno) );
    return -1;
  }
  
  bzero( &res, sizeof(res) );
  
#if 0
  {  /* Try to set it to make scan succeed, URB 26 */
     /* Definitly breaks */
    char data[10] = { 0x03, 0x00, 0x00, 0x00, 0x00, 0x3f, 0x04, 0x00, 0x00, 0xc3 };
    if (hp5400_command_write(iHandle, CMD_UNKNOWN, sizeof(data), data) < 0) {
      printf("failed to set unknown1\n");
      return -1;
    }
  }
#endif

  WarmupLamp( iHandle );
  
  {  /* Try to set it to make scan succeed, URB 53 */ /*  0x1B01 => 0x40  */
     /* I think this tries to cancel any existing scan */
    char flag = 0x40;
    if (hp5400_command_write(iHandle, CMD_USEGAMMA, sizeof(flag), &flag) < 0) {
      printf("failed to set gamma flag\n");
      return -1;
    }
  }

#if 1
  {  /* Try to set it to make scan succeed, URB 55  */
    char data[4] = { 0x02, 0x03, 0x03, 0x3C };
    if (hp5400_command_write(iHandle, CMD_UNKNOWN3, sizeof(data), data) < 0) {
      printf("failed to set unknown1\n");
      return -1;
    }
  }
#endif

#if 1
  {  /* Try to set it to make scan succeed, URB 59 */
    char flag = 0x04;
    if (hp5400_command_write(iHandle, CMD_UNKNOWN2, sizeof(flag), &flag) < 0) {
      printf("failed to set unknown2\n");
      return -1;
    }
  }
#endif

  { /* URB 67 */
    short dpi = htons( 300 );
    if (hp5400_command_write(iHandle, CMD_SETDPI, sizeof(dpi), &dpi) < 0) {
      printf("failed to set dpi\n");
      return -1;
    }
  }

  printf("Scan request: \n  ");
  {
    int i;
    for( i=0; i<sizeof(*req); i++ )
    {
      printf( "%02X ", ((unsigned char*)req)[i] );
    }
    printf("\n");
  }

  if (hp5400_command_write(iHandle, 0x2505, sizeof(*req), req) < 0) {
    printf("failed to send scan request\n");
    return -1;
  }
  
  {  /* Try to set it to make scan succeed, URB 71 */
    char flag = code;
    if (hp5400_command_write(iHandle, 0x1B05, sizeof(flag), &flag) < 0) {
      printf("failed to set gamma flag\n");
      return -1;
    }
  }

  if (hp5400_command_read(iHandle, CMD_SCANRESPONSE, sizeof(res), &res) < 0) {
    printf("failed to read scan response\n");
    return -1;
  }
  
  printf("Scan response: \n  ");
  {
    int i;
    for( i=0; i<sizeof(res); i++ )
    {
      printf( "%02X ", ((unsigned char*)&res)[i] );
    }
    printf("\n");
  }

  printf("Bytes to transfer: %d\nBitmap resolution: %d x %d\n", htonl(res.transfersize), htons(res.xsize), htons(res.ysize) );

  printf("Proceeding to scan\n");

  if( htonl(res.transfersize) == 0 )
  {
    printf("Hmm, size is zero. Obviously a problem. Aborting...\n");
    return -1;
  }    
  hp5400_bulk_read( iHandle, htonl( res.transfersize ) + 3*htons(res.ysize), 0xf000, file );

  {  /* Finish scan request */
    char flag = code;
    if (hp5400_command_write(iHandle, 0x1B01, sizeof(flag), &flag) < 0) {
      printf("failed to set gamma flag\n");
      return -1;
    }
  }

  fflush( file );
  
  rewind( file );
  {
    char *ppmfile;
    
    float pixels = ((float)htons(req->lenx)*(float)htons(req->leny)) *
                   ((float)htons(req->dpix)*(float)htons(req->dpiy)) /
                   (300.0*300.0);
    int bpp = rintf( (float)htonl(res.transfersize) / pixels );
    int planes = (bpp == 1)?1:3;
    bpp /= planes;
    
    ppmfile = malloc( strlen( filename ) +5 );
    strcpy( ppmfile, filename );
    strcat( ppmfile, ".ppm" );
    
    printf("Calc: (%d * %d) * (%d * %d) / (300 * 300) = %f\n", htons(req->lenx), htons(req->leny), htons(req->dpix), htons(req->dpiy), pixels );
    printf("Image: size=%d, pixels=%.0f, planes=%d, bpp=%d\n", htonl(res.transfersize), pixels, planes, bpp );
    if( bpp <= 3 )
      DecodeImage( file, planes, bpp, htons(res.xsize), htons(res.ysize), ppmfile );
    else
      printf("bpp has bogus value, not decoding\n");
  }
  
  fclose( file );
  return 0;
}

// The first calibration scan. Finds maximum of each CCD
int Calibrate1( int iHandle )
{
  struct ScanRequest req;
  bzero( &req, sizeof(req) );
  
  req.x1 = 0x08;
  req.dpix = htons( 300 );     /* = 300 dpi */
  req.dpiy = htons( 300 );     /* = 300 dpi */
  req.offx = htons( 0 );       /* = 0cm */
  req.offy = htons( 0 );       /* = 0cm */
  req.lenx = htons( 2690 );    /* = 22.78cm */
  req.leny = htons(   50 );    /* =  0.42cm */

  req.flags1 = htons( 0x0000 );
  req.flags2 = htons( 0x0010 );
  req.flags3 = htons( 0x3020 );   /* First calibration scan, 48bpp */
  
  req.gamma[0] = htons( 100 );
  req.gamma[1] = htons( 100 );
  req.gamma[2] = htons( 100 );

  return DoScan( iHandle, &req, "calibrate1.dat", 0x40 );
}

// The second calibration scan. Finds minimum of each CCD
int Calibrate2( int iHandle )
{
  struct ScanRequest req;
  bzero( &req, sizeof(req) );
  
  req.x1 = 0x08;
  req.dpix = htons( 300 );     /* = 300 dpi */
  req.dpiy = htons( 300 );     /* = 300 dpi */
  req.offx = htons( 0 );       /* = 0cm */
  req.offy = htons( 0 );       /* = 0cm */
  req.lenx = htons( 2690 );    /* = 22.78cm */
  req.leny = htons(   16 );    /* =  0.14cm */

  req.flags1 = htons( 0x0000 );
  req.flags2 = htons( 0x0010 );
  req.flags3 = htons( 0x3024 );   /* Second calibration scan, 48bpp */
  
  req.gamma[0] = htons( 100 );
  req.gamma[1] = htons( 100 );
  req.gamma[2] = htons( 100 );

  return DoScan( iHandle, &req, "calibrate2.dat", 0x00 );
}

int PreviewScan( int iHandle )
{
  struct ScanRequest req;
  bzero( &req, sizeof(req) );
    
  req.x1 = 0x08;
  req.dpix = htons( 75 );     /* = 75 dpi */
  req.dpiy = htons( 75 );     /* = 75 dpi */
  req.offx = htons( 0 );       /* = 0cm */
  req.offy = htons( 0 );       /* = 0cm */
  req.lenx = htons( 2552 );    /* = 21.61cm */
  req.leny = htons( 3510 );    /* = 29.72cm */

  req.flags1 = htons( 0x0080 );
  req.flags2 = htons( 0x0040 );
  req.flags3 = htons( 0x18E8 );   /* Try preview scan */

  req.gamma[0] = htons( 100 );
  req.gamma[1] = htons( 100 );
  req.gamma[2] = htons( 100 );

  SetGamma( iHandle );

  return DoScan( iHandle, &req, "output.dat", 0x40 );
}

char UISetup1[] = {
/* Offset 40 */
  0x50, 0x72, 0x6F, 0x63,  0x65, 0x73, 0x73, 0x69,  0x6E, 0x67, 0x14, 0x00,  0x52, 0x65, 0x61, 0x64,
  0x79, 0x00, 0x53, 0x63,  0x61, 0x6E, 0x6E, 0x65,  0x72, 0x20, 0x4C, 0x6F,  0x63, 0x6B, 0x65, 0x64,
  0x00, 0x45, 0x72, 0x72,  0x6F, 0x72, 0x00, 0x43,  0x61, 0x6E, 0x63, 0x65,  0x6C, 0x69, 0x6E, 0x67,
  0x14, 0x00, 0x50, 0x6F,  0x77, 0x65, 0x72, 0x53,  0x61, 0x76, 0x65, 0x20,  0x4F, 0x6E, 0x00, 0x53,
};

char UISetup2[] = {

  0x63, 0x61, 0x6E, 0x6E,  0x69, 0x6E, 0x67, 0x14,  0x00, 0x41, 0x44, 0x46,  0x20, 0x70, 0x61, 0x70, 
  0x65, 0x72, 0x20, 0x6A,  0x61, 0x6D, 0x00, 0x43,  0x6F, 0x70, 0x69, 0x65,  0x73, 0x00, 0x00,
};

int InitScanner( int iHandle )
{
  if( WriteByte( iHandle, 0xF200, 0x40 ) < 0 ) return -1;
  
  if (hp5400_command_write(iHandle, 0xF10B, sizeof( UISetup1 ), UISetup1 ) < 0) {       
    printf("failed to send UISetup1 (%d)\n", sizeof( UISetup1 ));
    return -1;
  }

  if( WriteByte( iHandle, 0xF200, 0x00 ) < 0 ) return -1;
  
  if (hp5400_command_write(iHandle, 0xF10C, sizeof( UISetup2 ), UISetup2 ) < 0) {       
    printf("failed to send UISetup2\n");
    return -1;
  }
  return 0;
}

int main(void)
{
  int  iHandle;
  char szVersion[32];

  assert( sizeof(struct ScanRequest) == 32 );
  assert( sizeof(struct ScanResponse) == 16 );
    
  printf("HP5400/5470C sample scan utility, by Martijn van Oosterhout <kleptog@svana.org>\n");
  printf("Based on the testutils by Bertrik Sikken (bertrik@zonnet.nl)\n");
  
  iHandle = hp5400_open();
  if (iHandle < 0) {
    printf("hp5400_open failed\n");
    exit(1);
  }

  /* read version info */
  if (hp5400_command_read(iHandle, CMD_GETVERSION, sizeof(szVersion), szVersion) < 0) {
    printf("failed to read version string\n");
    goto hp5400_close_exit;
  }
  
  printf("Version info string: \n  ");
  {
    int i;
    for( i=0; i<sizeof(szVersion); i++ )
    {
      printf( "%02X ", szVersion[i] );
    }
    for( i=0; i<sizeof(szVersion); i++ )
    {
      printf( "%c", ((szVersion[i]&0x60)==0)?'.':szVersion[i] );
    }
    printf("\n");
  }

  if( memcmp( szVersion+1, MatchVersion, sizeof(MatchVersion)-1 ) )
  {
    printf( "Sorry, unknown scanner version. Attempted match on '%s'\n", MatchVersion );
  }

  // This doesn't work for me, yet the Windows traces show this happening
  // InitScanner( iHandle );
  
  Calibrate1( iHandle );
  Calibrate2( iHandle );
  PreviewScan( iHandle );
  
hp5400_close_exit:
  hp5400_close(iHandle);
  exit(0);
}

