/*
  (c) 2003 Martijn van Oosterhout, kleptog@svana.org
  (c) 2002 Bertrik Sikken, bertrik@zonnet.nl

  Transport layer for communication with HP5400/5470 scanner.

  Implementation using the Linux 'scanner' driver module
  
  Additions to support bulk data transport. Added debugging info - 19/02/2003 Martijn
*/

#include <stdio.h>        /* fprintf */
#include <unistd.h>       /* open */
#include <fcntl.h>
#include <stdlib.h>

#include "hp5400_xfer.h"

#include <asm/types.h> /* types used in ioctls */
#include <sys/ioctl.h> /* ioctl */
#include <linux/usb.h> /* linux usb ioctls */

#ifdef STANDALONE
#define DBG fprintf
#define DBG_MSG stdout
#define DBG_ERR stderr
#endif

#define CMD_INITBULK1   0x0087    /* send 0x14 */
#define CMD_INITBULK2   0x0083    /* send 0x24 */
#define CMD_INITBULK3   0x0082    /* transfer length 0xf000 */

/* copied from usb.h */
#define SCANNER_IOCTL_VENDOR _IOR('U', 0x20, int)
#define SCANNER_IOCTL_PRODUCT _IOR('U', 0x21, int)
#define SCANNER_IOCTL_CTRLMSG _IOWR('U', 0x22, devrequest)

typedef struct {
  __u8 requesttype;
  __u8 request;
  __u16 value;
  __u16 index;
  __u16 length;
} devrequest __attribute__ ((packed));


static void
_UsbWriteControl(int fd, int iValue, int iIndex, void *pabData, int iSize)
{
  struct ctrlmsg_ioctl {
    devrequest  req;
    void        *data;
  } cmsg;

  cmsg.req.requesttype  = USB_TYPE_VENDOR | USB_RECIP_DEVICE | USB_DIR_OUT;
  cmsg.req.request      = (iSize > 1) ? 0x04 : 0x0C;
  cmsg.req.value        = iValue;
  cmsg.req.index        = iIndex;
  cmsg.req.length       = iSize;
  cmsg.data             = pabData;

  DBG(DBG_MSG, "Write: reqtype = 0x%02X, req = 0x%02X, value = %04X, len = %d\n", cmsg.req.requesttype, cmsg.req.request,cmsg.req.value, iSize);

  if( iSize > 0 )
  {
    int i;
    DBG(DBG_MSG, "  Data: " );
    for( i=0; i<iSize && i<8; i++ )
      DBG(DBG_MSG, "%02X ", ((unsigned char*)pabData)[i]);
    if( iSize > 8 )
      DBG(DBG_MSG, "..." );
    DBG(DBG_MSG, "\n" );
  }
  
  if (fd != -1) {
    ioctl(fd, SCANNER_IOCTL_CTRLMSG, &cmsg);
  }
}

void hp5400_command_write_noverify( int fd, int iValue, void *pabData, int iSize)
{  
  _UsbWriteControl(fd, iValue, 0, pabData, iSize);
}

static void
_UsbReadControl(int fd, int iValue, int iIndex, void *pabData, int iSize)
{
  struct ctrlmsg_ioctl {
    devrequest  req;
    void        *data;
  } cmsg;

  cmsg.req.requesttype  = USB_TYPE_VENDOR | USB_RECIP_DEVICE | USB_DIR_IN;
  cmsg.req.request      = (iSize > 1) ? 0x04 : 0x0C;
  cmsg.req.value        = iValue;
  cmsg.req.index        = iIndex;
  cmsg.req.length       = iSize;
  cmsg.data             = pabData;

  DBG(DBG_MSG, "Read: reqtype = 0x%02X, req = 0x%02X, value = %04X\n", cmsg.req.requesttype, cmsg.req.request,cmsg.req.value);

  if (fd != -1) {
    ioctl(fd, SCANNER_IOCTL_CTRLMSG, &cmsg);
  }
}



int hp5400_open(void)
{
  int fd, iVendor, iProduct;

  fd = open("/dev/usb/scanner0", O_RDWR);
  if (fd < 0) {
    /* try /dev/usbscanner */
    fd = open("/dev/usbscanner", O_RDWR);
    if (fd < 0) {
      return fd;
    }
  }

  iVendor = 0;
  iProduct = 0;
  ioctl(fd, SCANNER_IOCTL_VENDOR, &iVendor);
  ioctl(fd, SCANNER_IOCTL_PRODUCT, &iProduct);

  DBG(DBG_MSG, "vendor/product 0x%04X-0x%04X\n", iVendor, iProduct);

  return fd;
}


void hp5400_close(int iHandle)
{
  close(iHandle);
}


/* returns value > 0 if verify ok */
int hp5400_command_verify(int iHandle, int iCmd)
{
  unsigned char abData[4];
  int fd;

  if (iHandle < 0) {
    DBG(DBG_ERR, "hp5400_command_verify: invalid handle\n");
    return -1;
  }
  fd = iHandle;

  /* command 0xc500: read back previous command */
  _UsbReadControl(fd, 0xc500, 0, (char *)abData, 2);

  if (abData[0] != (iCmd >> 8)) {
    DBG(DBG_ERR, "hp5400_command_verify failed, expected 0x%02X%02X, got 0x%02X%02X\n",
      (int)(iCmd >> 8), (int)(iCmd & 0xff), (int)abData[0], (int)abData[1]);
      
    return -1;
  }

  if( abData[1] != 0 )   // Error code non-zero
  {
    _UsbReadControl(fd, 0x0300, 0, (char *)abData, 3);
    DBG(DBG_ERR, "  error response is: %02X %02X %02X\n", abData[0], abData[1], abData[2]);
    
    return -1;
  }
  
  DBG(DBG_MSG, "Command %02X verified\n", abData[0] );
  return 1;
}


/* returns > 0 if command OK */
int hp5400_command_read(int iHandle, int iCmd, int iLen, void *pbData)
{
  int fd;

  if (iHandle < 0) {
    DBG(DBG_ERR, "hp5400_command_read: invalid handle\n");
    return -1;
  }
  fd = iHandle;

  _UsbReadControl(fd, iCmd, 0, pbData, iLen);

  return hp5400_command_verify(iHandle, iCmd);
}


/* returns >0 if command OK */
int hp5400_command_write(int iHandle, int iCmd, int iLen, void *pbData)
{
  int fd;

  if (iHandle < 0) {
    DBG(DBG_ERR, "hp5400_command_write: invalid handle\n");
    return -1;
  }
  fd = iHandle;

  _UsbWriteControl(fd, iCmd, 0, (char *)pbData, iLen);

  return hp5400_command_verify(iHandle, iCmd);
}

/* returns >0 if command OK */
int hp5400_bulk_read(int iHandle, int len, int block, FILE *file )
{
  int fd;
  char x1 = 0x14, x2 = 0x24;
  short buf[4] = { 0, 0, block, 0 };
  unsigned char * buffer;
  int res = 0;
  
  if (iHandle < 0) {
    DBG(DBG_ERR, "hp5400_command_write: invalid handle\n");
    return -1;
  }
  fd = iHandle;
  
  buffer = malloc( block );

  _UsbWriteControl(fd, CMD_INITBULK1, 0, &x1, 1);
  _UsbWriteControl(fd, CMD_INITBULK2, 0, &x2, 1);

  while( len > 0 )
  {
    _UsbWriteControl(fd, CMD_INITBULK3, 0, (unsigned char*)&buf, sizeof(buf) );
    res = read( fd, buffer, block );
    DBG(DBG_MSG,"Read returned %d, %d remain\n", res, len );
    if( res > 0 )
    {
      fwrite( buffer, (len < res)?len:res, 1, file );
    }
    len -= block;
  }
  
  return 0;
}

/* returns >0 if command OK */
int hp5400_bulk_read_block(int iHandle, int iCmd, void *cmd, int cmdlen, void *buffer, int len )
{
  int fd;
  int res = 0;
  
  if (iHandle < 0) {
    DBG(DBG_ERR, "hp5400_command_write: invalid handle\n");
    return -1;
  }
  fd = iHandle;
  
  _UsbWriteControl(fd, iCmd, 0, cmd, cmdlen );
  res = read( fd, buffer, len );
  DBG(DBG_MSG,"Read returned %d, %d remain\n", res, len );
  return res;
}

/* returns >0 if command OK */
int hp5400_bulk_command_write(int iHandle, int iCmd, void *cmd, int cmdlen, int datalen, int block, char *data )
{
  int fd;
  int res = 0, offset = 0;
  
  if (iHandle < 0) {
    DBG(DBG_ERR, "hp5400_command_write: invalid handle\n");
    return -1;
  }
  fd = iHandle;

  DBG(DBG_MSG,"bulk_command_write(%04X,<%d bytes>,<%d bytes>)\n", iCmd, cmdlen, datalen);
    
  _UsbWriteControl(fd, iCmd, 0, cmd, cmdlen);

  while( datalen > 0 )
  {
    {
      int i;
      DBG(DBG_MSG, "  Data: " );
      for( i=0; i<datalen && i<block && i<8; i++ )
        DBG(DBG_MSG, "%02X ", ((unsigned char*)data+offset)[i]);
      if( i >= 8 )
        DBG(DBG_MSG, "..." );
      DBG(DBG_MSG, "\n" );
    }
    
    res = write( fd, data+offset, (datalen < block)?datalen:block );
    DBG(DBG_MSG,"Write returned %d, %d remain\n", res, datalen );
    datalen -= block;
    offset  += block;
  }
  
  return hp5400_command_verify( iHandle, iCmd );
}

