/*
  Copyright (C) 2003 Martijn van Oosterhout <kleptog@svana.org>
  
  This file was initially copied from the hp3300 testools and adjusted to
  suit. Original copyright notice follows:
  
  Copyright (C) 2001 Bertrik Sikken (bertrik@zonnet.nl)

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/

/*
    SANE interface for hp54xx scanners. Prototype.
    Parts of this source were inspired by other backends.
*/

/* definitions for debug */
#define BACKEND_NAME hp5400
#define BUILD 1

#include "sane/sane.h"
//#include "sane/config.h"
#include "sane/sanei.h"
#include "sane/sanei_backend.h"
#include "sane/sanei_config.h"
#include "sane/saneopts.h"

#include <stdlib.h>         /* malloc, free */
#include <string.h>         /* memcpy */
#include <stdio.h>

#define DBG_ASSERT  1
#define DBG_ERR     16
#define DBG_MSG     32

#define HP5400_CONFIG_FILE "hp5400.conf"

#include "hp5400.h"

//#include "hp5400_xfer_cfg.h"

/* (source) includes for data transfer methods */
#include "hp5400.c"
#include "hp5400_scanner.c"
/* #include "hp5400_saneiusb.c" */
#if defined(LINUX_USB_SUPPORT)
  #include "hp5400_linux.c"
#endif
#if defined(USCANNER_SUPPORT)
  #include "hp5400_uscanner.c"
#endif
#if defined(LIBUSB_SUPPORT)
  #include "hp5400_libusb.c"
#endif
#if defined(LIBIEEE1284_SUPPORT)
  #include "hp5400_ieee1284.c"
#endif


#define ASSERT(cond) (!(cond) ? DBG(DBG_ASSERT, "!!! ASSERT(%S) FAILED!!!\n",STRINGIFY(cond));)

/* other definitions */
#define TRUE 1
#define FALSE 0

#define MM_TO_PIXEL(_mm_, _dpi_)    ((_mm_) * (_dpi_) / 25.4)
#define PIXEL_TO_MM(_pixel_, _dpi_) ((_pixel_) * 25.4 / (_dpi_))

#define NUM_GAMMA_ENTRIES  65536
#define NUM_CAL_ENTRIES    (2690 * 3 * 2)    /* 16140 */

/* Device filename for USB access */
char * usb_devfile = "/dev/usb/scanner0";

/* options enumerator */
typedef enum {
  optCount = 0,

  optGroupGeometry,
  optTLX, optTLY, optBRX, optBRY,
  optDPI,

  optGroupImage,

  optGammaTableRed,     /* Gamma Tables */
  optGammaTableGreen,
  optGammaTableBlue,

  optLast,
/* put temporarily disabled options here after optLast */
/*  
  optGroupMisc,
  optLamp,
*/
//  optCalibrate,
//  optGamma             /* analog gamma = single number */

} EOptionIndex;


typedef union {
  SANE_Word w;
  SANE_Word *wa;              /* word array */
  SANE_String s;
} TOptionValue;


typedef struct {
  SANE_Option_Descriptor  aOptions[optLast];
  TOptionValue            aValues[optLast];

  TScanParams             ScanParams;
  THWParams               HWParams;

  TDataPipe               DataPipe;
  int                     iLinesLeft;

  SANE_Int                *aGammaTableR;    /* a 16-to-16 bit color lookup table */
  SANE_Int                *aGammaTableG;    /* a 16-to-16 bit color lookup table */
  SANE_Int                *aGammaTableB;    /* a 16-to-16 bit color lookup table */
  
  SANE_Int                *aCalibrationTable;   /* 2690 sets of (minR,minG,minB,maxR,maxG,maxB) */

  int                     fScanning;            /* TRUE if actively scanning */
  int                     fCanceled;
} TScanner;


/* linked list of SANE_Device structures */
typedef struct TDevListEntry {
  struct TDevListEntry *pNext;
  SANE_Device dev;
} TDevListEntry;

static TDevListEntry *_pFirstSaneDev = 0;
static int iNumSaneDev = 0;
static const SANE_Device **_pSaneDevList = 0;


/* option constraints */
static const SANE_Range rangeGammaTable = {0, 65535, 1};
static const SANE_Int   setResolutions[] = {3, 150, 300, 600};
static const SANE_Range rangeGamma = {SANE_FIX(0.25), SANE_FIX(4.0),
 SANE_FIX(0.0)};
static const SANE_Range rangeXmm = {0, 220, 1};
static const SANE_Range rangeYmm = {0, 290, 1};


static void _InitOptions(TScanner *s)
{
  int i, j;
  SANE_Option_Descriptor *pDesc;
  TOptionValue *pVal;

  /* set a neutral gamma */
  if( s->aGammaTableR == NULL )   /* Not yet allocated */
  {
    s->aGammaTableR = malloc( NUM_GAMMA_ENTRIES * sizeof( SANE_Int ) );
    s->aGammaTableG = malloc( NUM_GAMMA_ENTRIES * sizeof( SANE_Int ) );
    s->aGammaTableB = malloc( NUM_GAMMA_ENTRIES * sizeof( SANE_Int ) );

    for (j = 0; j < NUM_GAMMA_ENTRIES; j++) {
      s->aGammaTableR[j] = j;
      s->aGammaTableG[j] = j;
      s->aGammaTableB[j] = j;
    }
  }
  
  for (i = optCount; i < optLast; i++) {

    pDesc = &s->aOptions[i];
    pVal = &s->aValues[i];

    /* defaults */
    pDesc->name   = "";
    pDesc->title  = "";
    pDesc->desc   = "";
    pDesc->type   = SANE_TYPE_INT;
    pDesc->unit   = SANE_UNIT_NONE;
    pDesc->size   = sizeof(SANE_Word);
    pDesc->constraint_type = SANE_CONSTRAINT_NONE;
    pDesc->cap    = 0;

    switch (i) {

    case optCount:
      pDesc->title  = SANE_TITLE_NUM_OPTIONS;
      pDesc->desc   = SANE_DESC_NUM_OPTIONS;
      pDesc->cap    = SANE_CAP_SOFT_DETECT;
      pVal->w       = (SANE_Word)optLast;
      break;

    case optGroupGeometry:
      pDesc->title  = "Geometry";
      pDesc->type   = SANE_TYPE_GROUP;
      pDesc->size   = 0;
      break;

    case optTLX:
      pDesc->name   = SANE_NAME_SCAN_TL_X;
      pDesc->title  = SANE_TITLE_SCAN_TL_X;
      pDesc->desc   = SANE_DESC_SCAN_TL_X;
      pDesc->unit   = SANE_UNIT_MM;
      pDesc->constraint_type  = SANE_CONSTRAINT_RANGE;
      pDesc->constraint.range = &rangeXmm;
      pDesc->cap    = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
      pVal->w       = rangeXmm.min;
      break;

    case optTLY:
      pDesc->name   = SANE_NAME_SCAN_TL_Y;
      pDesc->title  = SANE_TITLE_SCAN_TL_Y;
      pDesc->desc   = SANE_DESC_SCAN_TL_Y;
      pDesc->unit   = SANE_UNIT_MM;
      pDesc->constraint_type  = SANE_CONSTRAINT_RANGE;
      pDesc->constraint.range = &rangeYmm;
      pDesc->cap    = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
      pVal->w       = rangeYmm.min;
      break;

    case optBRX:
      pDesc->name   = SANE_NAME_SCAN_BR_X;
      pDesc->title  = SANE_TITLE_SCAN_BR_X;
      pDesc->desc   = SANE_DESC_SCAN_BR_X;
      pDesc->unit   = SANE_UNIT_MM;
      pDesc->constraint_type  = SANE_CONSTRAINT_RANGE;
      pDesc->constraint.range = &rangeXmm;
      pDesc->cap    = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
      pVal->w       = rangeXmm.max;
      break;

    case optBRY:
      pDesc->name   = SANE_NAME_SCAN_BR_Y;
      pDesc->title  = SANE_TITLE_SCAN_BR_Y;
      pDesc->desc   = SANE_DESC_SCAN_BR_Y;
      pDesc->unit   = SANE_UNIT_MM;
      pDesc->constraint_type  = SANE_CONSTRAINT_RANGE;
      pDesc->constraint.range = &rangeYmm;
      pDesc->cap    = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
      pVal->w       = rangeYmm.max;
      break;

    case optDPI:
      pDesc->name   = SANE_NAME_SCAN_RESOLUTION;
      pDesc->title  = SANE_TITLE_SCAN_RESOLUTION;
      pDesc->desc   = SANE_DESC_SCAN_RESOLUTION;
      pDesc->unit   = SANE_UNIT_DPI;
      pDesc->constraint_type  = SANE_CONSTRAINT_WORD_LIST;
      pDesc->constraint.word_list = setResolutions;
      pDesc->cap    = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
      pVal->w       = setResolutions[1];
      break;

    case optGroupImage:
      pDesc->title  = SANE_I18N("Image");
      pDesc->type   = SANE_TYPE_GROUP;
      pDesc->size   = 0;
      break;

    case optGammaTableRed:
      pDesc->name   = SANE_NAME_GAMMA_VECTOR_R;
      pDesc->title  = SANE_TITLE_GAMMA_VECTOR_R;
      pDesc->desc   = SANE_DESC_GAMMA_VECTOR_R;
      pDesc->size   = NUM_GAMMA_ENTRIES;
      pDesc->constraint_type = SANE_CONSTRAINT_RANGE;
      pDesc->constraint.range = &rangeGammaTable;
      pDesc->cap    = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
      pVal->wa      = s->aGammaTableR;
      break;

    case optGammaTableGreen:
      pDesc->name   = SANE_NAME_GAMMA_VECTOR_G;
      pDesc->title  = SANE_TITLE_GAMMA_VECTOR_G;
      pDesc->desc   = SANE_DESC_GAMMA_VECTOR_G;
      pDesc->size   = NUM_GAMMA_ENTRIES;
      pDesc->constraint_type = SANE_CONSTRAINT_RANGE;
      pDesc->constraint.range = &rangeGammaTable;
      pDesc->cap    = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
      pVal->wa      = s->aGammaTableG;
      break;

    case optGammaTableBlue:
      pDesc->name   = SANE_NAME_GAMMA_VECTOR_B;
      pDesc->title  = SANE_TITLE_GAMMA_VECTOR_B;
      pDesc->desc   = SANE_DESC_GAMMA_VECTOR_B;
      pDesc->size   = NUM_GAMMA_ENTRIES;
      pDesc->constraint_type = SANE_CONSTRAINT_RANGE;
      pDesc->constraint.range = &rangeGammaTable;
      pDesc->cap    = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
      pVal->wa      = s->aGammaTableB;
      break;
#if 0
    case optGroupMisc:
      pDesc->title  = SANE_I18N("Miscellaneous");
      pDesc->type   = SANE_TYPE_GROUP;
      pDesc->size   = 0;
      break;
#endif
#if 0
    case optLamp:
      pDesc->name   = "lamp";
      pDesc->title  = SANE_I18N("Lamp status");
      pDesc->desc   = SANE_I18N("Switches the lamp on or off.");
      pDesc->type   = SANE_TYPE_BOOL;
      pDesc->cap    = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
      /* switch the lamp on when starting for first the time */
      pVal->w       = SANE_TRUE;
      break;
#endif
#if 0
    case optCalibrate:
      pDesc->name   = "calibrate";
      pDesc->title  = SANE_I18N("Calibrate");
      pDesc->desc   = SANE_I18N("Calibrates for black and white level.");
      pDesc->type   = SANE_TYPE_BUTTON;
      pDesc->cap    = SANE_CAP_SOFT_SELECT;
      pDesc->size   = 0;
      break;
#endif
    default:
      DBG(DBG_ERR, "Uninitialised option %d\n", i);
      break;
    }
  }
}


static int _ReportDevice(TScannerModel *pModel, char *pszDeviceName)
{
  TDevListEntry *pNew, *pDev;

  DBG(DBG_MSG, "hp5400: _ReportDevice '%s'\n", pszDeviceName);

  pNew = malloc(sizeof(TDevListEntry));
  if (!pNew) {
    DBG(DBG_ERR, "no mem\n");
    return -1;
  }
  
  /* add new element to the end of the list */
  if (_pFirstSaneDev == NULL) {
    _pFirstSaneDev = pNew;
  }
  else {
    for (pDev = _pFirstSaneDev; pDev->pNext; pDev = pDev->pNext) {
      ;
    }
    pDev->pNext = pNew;
  }

  /* fill in new element */
  pNew->pNext = 0;
  pNew->dev.name = strdup(pszDeviceName);
  pNew->dev.vendor = pModel->pszVendor;
  pNew->dev.model = pModel->pszName;
  pNew->dev.type = "flatbed scanner";

  iNumSaneDev++;

  return 0;
}

/*****************************************************************************/

SANE_Status
sane_init(SANE_Int *piVersion, SANE_Auth_Callback pfnAuth)
{
  FILE * conf_fp; /* Config file stream  */
  SANE_Char line[PATH_MAX];
  SANE_Char * str = NULL;
  SANE_String_Const proper_str;
  int nline = 0;

  /* prevent compiler from complaing about unused parameters */
  pfnAuth = pfnAuth;

  DBG_INIT();
  DBG(DBG_MSG, "sane_init\n");

  /* install transfer methods */
#if defined(SANEI_USB_SUPPORT)
  HP5400XferRegisterModule(&SaneiUsbDev);
#endif
#if defined(LINUX_USB_SUPPORT)
  HP5400XferRegisterModule(&LnxUsbDev);
#endif
#if defined(USCANNER_SUPPORT)
  HP5400XferRegisterModule(&uscannerDev);
#endif
#if defined(LIBUSB_SUPPORT)
  HP5400XferRegisterModule(&LibUsbDev);
#endif
#if defined(LIBIEEE1284_SUPPORT)
  HP5400XferRegisterModule(&IEEE1284Dev);
#endif

  conf_fp = sanei_config_open (HP5400_CONFIG_FILE);

  if (conf_fp) {
    DBG(DBG_MSG, "Reading config file\n");

    while (sanei_config_read (line, sizeof (line), conf_fp)) {
      ++nline;

      if (str) {
        free(str);
      }

      proper_str = sanei_config_get_string (line, &str);

      /* Discards white lines and comments */
      if (!str || proper_str == line || str[0] == '#' ) {
        DBG(DBG_MSG, "Discarding line %d\n", nline);
      }
      else {
        /* If line's not blank or a comment, then it's the device
         * filename. usb_devfile is resized to hold the device filename string  */
        usb_devfile = malloc ( sizeof (line) * sizeof(SANE_Char) );
        if (usb_devfile == NULL) {
          DBG(DBG_ERR, "Unable to allocate memory for device filename string\n");
          DBG(DBG_MSG, "Using default built-in values\n");
          break;
        }
        strcpy (usb_devfile, line);
        DBG(DBG_MSG, "Using %s as device file\n", usb_devfile);
      }
    } /* while */
    fclose(conf_fp);
  }
  else {
    DBG (DBG_ERR, "Unable to read config file \"%s\": %s\n", HP5400_CONFIG_FILE,
        strerror(errno));
    DBG (DBG_MSG, "Using default built-in values\n");
  }

  if (piVersion != NULL) {
    *piVersion = SANE_VERSION_CODE (V_MAJOR, V_MINOR, BUILD);
  }

  /* initialise transfer methods */
  iNumSaneDev = 0;
  HP5400Detect( usb_devfile, _ReportDevice );

  return SANE_STATUS_GOOD;
}


void
sane_exit(void)
{
  TDevListEntry *pDev, *pNext;

  DBG(DBG_MSG, "sane_exit\n");

  /* free device list memory */
  if (_pSaneDevList) {
    for (pDev = _pFirstSaneDev; pDev; pDev = pNext) {
      pNext = pDev->pNext;
      free((void *)pDev->dev.name);
      free(pDev);
    }
    _pFirstSaneDev = 0;
    free(_pSaneDevList);
    _pSaneDevList = 0;
  }
}


SANE_Status
sane_get_devices(const SANE_Device ***device_list, SANE_Bool local_only)
{
  TDevListEntry *pDev;
  int i;

  DBG(DBG_MSG, "sane_get_devices\n");

  local_only = local_only;

  if (_pSaneDevList) {
    free(_pSaneDevList);
  }

  _pSaneDevList = malloc(sizeof(*_pSaneDevList) * (iNumSaneDev + 1));
  if (!_pSaneDevList) {
    DBG(DBG_MSG, "no mem\n");
    return SANE_STATUS_NO_MEM;
  }
  i = 0;
  for (pDev = _pFirstSaneDev; pDev; pDev = pDev->pNext) {
    _pSaneDevList[i++] = &pDev->dev;
  }
  _pSaneDevList[i++] = 0; /* last entry is 0 */

  *device_list = _pSaneDevList;

  return SANE_STATUS_GOOD;
}


SANE_Status
sane_open(SANE_String_Const name, SANE_Handle *h)
{
  TScanner *s;

  DBG(DBG_MSG, "sane_open: %s\n", name);

  /* check the name */
  if (strlen(name) == 0) {
    /* default to first available device */
    name = _pFirstSaneDev->dev.name;
  }

  s = malloc(sizeof(TScanner));
  if (!s) {
    DBG(DBG_MSG, "malloc failed\n");
    return SANE_STATUS_NO_MEM;
  }
  
  if (HP5400Open(&s->HWParams, (char *)name) < 0) {
    /* is this OK ? */
    DBG(DBG_ERR, "HP5400Open failed\n");
    free((void *)s);
    return SANE_STATUS_INVAL; /* is this OK? */
  }
  memset( s, 0, sizeof(TScanner) );   /* Clear everything to zero */
  _InitOptions(s);
  *h = s;

  /* Turn on lamp by default at startup */
//  SetLamp(&s->HWParams, TRUE);

  return SANE_STATUS_GOOD;
}


void
sane_close(SANE_Handle h)
{
  TScanner *s;

  DBG(DBG_MSG, "sane_close\n");

  s = (TScanner *)h;

  /* turn of scanner lamp */
  SetLamp(&s->HWParams, FALSE);

  /* close scanner */
  HP5400Close(&s->HWParams);

  /* free scanner object memory */
  free((void *)s);
}


const SANE_Option_Descriptor *
sane_get_option_descriptor(SANE_Handle h, SANE_Int n)
{
  TScanner *s;

  DBG(DBG_MSG, "sane_get_option_descriptor %d\n", n);

  if ((n < optCount) || (n >= optLast)) {
    return NULL;
  }

  s = (TScanner *)h;
  return &s->aOptions[n];
}


SANE_Status
sane_control_option(SANE_Handle h, SANE_Int n, SANE_Action Action, void *pVal,
                    SANE_Int *pInfo)
{
  TScanner    *s;
//  int         fVal;
//  static char szTable[15000];
//  char        szTemp[16];
//  int         *pi;
//  int         i;
  SANE_Int    info;
//  int        fLampIsOn;

  DBG(DBG_MSG, "sane_control_option: option %d, action %d\n", n, Action);

  s = (TScanner *)h;
  info = 0;

  switch (Action) {
  case SANE_ACTION_GET_VALUE:
    switch (n) {

    /* Get options of type SANE_Word */
    case optCount:
    case optDPI:
//    case optGamma:
    case optTLX: case optTLY: case optBRX: case optBRY:
      DBG(DBG_MSG, "sane_control_option: SANE_ACTION_GET_VALUE %d = %d\n", n,
          (int)s->aValues[n].w);
      *(SANE_Word*)pVal = s->aValues[n].w;
      break;

    /* Get options of type SANE_Word array */
    case optGammaTableRed:
    case optGammaTableGreen:
    case optGammaTableBlue:
      DBG(DBG_MSG, "Reading red gamma table\n");
      memcpy(pVal, s->aValues[n].wa, s->aOptions[n].size);
      break;
#if 0
    /* Get options of type SANE_Bool */
    case optLamp:
      GetLamp(&s->HWParams, &fLampIsOn);
      *(SANE_Bool *)pVal = fLampIsOn;
      break;
#endif
#if 0
    case optCalibrate:
      /*  although this option has nothing to read,
          it's added here to avoid a warning when running scanimage --help */
      break;
#endif
    default:
      DBG(DBG_MSG, "SANE_ACTION_GET_VALUE: Invalid option (%d)\n", n);
    }
    break;


  case SANE_ACTION_SET_VALUE:
    if (s->fScanning) {
      DBG(DBG_ERR, "sane_control_option: SANE_ACTION_SET_VALUE not allowed during scan\n");
      return SANE_STATUS_INVAL;
    }
    switch (n) {

    case optCount:
      return SANE_STATUS_INVAL;
      break;

    case optDPI: case optTLX: case optTLY: case optBRX: case optBRY:
      info |= SANE_INFO_RELOAD_PARAMS;
      /* fall through */
/*
    case optGamma:
      DBG(DBG_MSG, "sane_control_option: SANE_ACTION_SET_VALUE %d = %d\n", n,
          (int)s->aValues[n].w);
      s->aValues[n].w = *(SANE_Word *)pVal;
      break;
*/
    case optGammaTableRed:
    case optGammaTableGreen:
    case optGammaTableBlue:
      DBG(DBG_MSG, "Writing red gamma table\n");
/*
      pi = (SANE_Int *)pVal;
      strcpy(szTable, "");
      for (i = 0; i < 4096; i++) {
        if ((i % 32) == 0) {
          sprintf(szTemp, " %04X", pi[i]);
          strcat(szTable, szTemp);
        }
      }
*/
      memcpy(s->aValues[n].wa, pVal, s->aOptions[n].size);
//      DBG(DBG_MSG, "Gamma table summary:\n%s\n", szTable);
      break;
/*
    case optLamp:
      fVal = *(SANE_Bool *)pVal;
      DBG(DBG_MSG, "lamp %s\n", fVal ? "on" : "off");
      SetLamp(&s->HWParams, fVal);
      break;
*/
#if 0
    case optCalibrate:
/*       SimpleCalib(&s->HWParams); */
      break;
#endif
    default:
      DBG(DBG_ERR, "SANE_ACTION_SET_VALUE: Invalid option (%d)\n", n);
    }
    if (pInfo != NULL) {
      *pInfo = info;
    }
    break;


  case SANE_ACTION_SET_AUTO:
    return SANE_STATUS_UNSUPPORTED;


  default:
    DBG(DBG_ERR, "Invalid action (%d)\n", Action);
    return SANE_STATUS_INVAL;
  }

  return SANE_STATUS_GOOD;
}



SANE_Status
sane_get_parameters(SANE_Handle h, SANE_Parameters *p)
{
  TScanner *s;
  DBG(DBG_MSG, "sane_get_parameters\n");

  s = (TScanner *)h;

  /* first do some checks */
  if (s->aValues[optTLX].w >= s->aValues[optBRX].w) {
    DBG(DBG_ERR, "TLX should be smaller than BRX\n");
    return SANE_STATUS_INVAL; /* proper error code? */
  }
  if (s->aValues[optTLY].w >= s->aValues[optBRY].w) {
    DBG(DBG_ERR, "TLY should be smaller than BRY\n");
    return SANE_STATUS_INVAL; /* proper error code? */
  }

  /* return the data */
  p->format = SANE_FRAME_RGB;
  p->last_frame = SANE_TRUE;

  p->lines = MM_TO_PIXEL(s->aValues[optBRY].w - s->aValues[optTLY].w,
                         s->aValues[optDPI].w);
  p->depth = 8;
  p->pixels_per_line = MM_TO_PIXEL(s->aValues[optBRX].w - s->aValues[optTLX].w,
                                   s->aValues[optDPI].w);
  p->bytes_per_line = p->pixels_per_line * 3;

  return SANE_STATUS_GOOD;
}


SANE_Status
sane_start(SANE_Handle h)
{
  TScanner        *s;
  SANE_Parameters par;
  int             iLineCorr;
//  static byte     abGamma[4096];
//  static byte     abCalibTable[HW_PIXELS * 6];

  DBG(DBG_MSG, "sane_start\n");

  s = (TScanner *)h;

  if (sane_get_parameters(h, &par) != SANE_STATUS_GOOD) {
    DBG(DBG_MSG, "Invalid scan parameters\n");
    return SANE_STATUS_INVAL;
  }
  s->iLinesLeft = par.lines;

  /* fill in the scanparams using the option values */
  s->ScanParams.iDpi = s->aValues[optDPI].w;
  s->ScanParams.iLpi = s->aValues[optDPI].w;
#if 0
  /* calculate correction for filling of circular buffer */
  iLineCorr = 3 * s->HWParams.iSensorSkew;     /* usually 16 motor steps */
  /* calculate correction for garbage lines */
  iLineCorr += s->HWParams.iSkipLines * (HW_LPI / s->ScanParams.iLpi);
#endif
  s->ScanParams.iTop =
    MM_TO_PIXEL(s->aValues[optTLY].w + s->HWParams.iTopLeftY, HW_LPI) - iLineCorr;
  s->ScanParams.iLeft =
    MM_TO_PIXEL(s->aValues[optTLX].w + s->HWParams.iTopLeftX, HW_DPI);

  s->ScanParams.iWidth = par.pixels_per_line;
  s->ScanParams.iHeight = par.lines;
  s->ScanParams.iBottom = 14200UL;
//  s->ScanParams.fCalib = FALSE;

  /* perform a simple calibration just before scanning */
/*  SimpleCalib(&s->HWParams, abCalibTable); */

  /* copy gamma table */
/*
  for (i = 0; i < 4096; i++) {
    abGamma[i] = s->aValues[optGammaTable].wa[i];
  }
  WriteGammaCalibTable(abGamma, abGamma, abGamma, abCalibTable, 0, 0, &s->HWParams);
*/

  /* prepare the actual scan */
  if (!InitScan(&s->ScanParams, &s->HWParams)) {
    DBG(DBG_MSG, "Invalid scan parameters\n");
    return SANE_STATUS_INVAL;
  }

#if 0
  /* init data pipe */
  s->DataPipe.iSkipLines = s->HWParams.iSkipLines;
  s->DataPipe.iBytesLeft = 0;
  /* hack */
  s->DataPipe.pabLineBuf = (byte *)malloc(HW_PIXELS * 3);
#endif

  CircBufferInit( s->HWParams.iXferHandle, &s->DataPipe, par.bytes_per_line,
                  1 /*bpp*/, 1 /* s->ScanParams.iLpi * s->HWParams.iSensorSkew / HW_LPI */, 0 /* s->HWParams.iReversedHead */);

  s->fScanning = TRUE;
  return SANE_STATUS_GOOD;
}


SANE_Status
sane_read(SANE_Handle h, SANE_Byte *buf, SANE_Int maxlen, SANE_Int *len)
{
  TScanner  *s;
  TDataPipe *p;

  DBG(DBG_MSG, "sane_read: buf=%p, maxlen=%d, ", buf, maxlen);

  s = (TScanner *)h;

  /* sane_read only allowed after sane_start */
  if (!s->fScanning) {
    DBG(DBG_ERR, "sane_read: sane_read only allowed after sane_start\n");
    return SANE_STATUS_INVAL;
  }

  p = &s->DataPipe;

#if 0
  /* anything left to read? */
  if ((s->iLinesLeft == 0) && (p->iBytesLeft == 0)) {
    CircBufferExit(&s->DataPipe);
    free(p->pabLineBuf);
    FinishScan(&s->HWParams);
    *len = 0;
    DBG(DBG_MSG, "sane_read: end of scan\n");
    s->fScanning = FALSE;
    return SANE_STATUS_EOF;
  }

  /* time to read the next line? */
  if (p->iBytesLeft == 0) {
    CircBufferGetLine(s->HWParams.iXferHandle, p, p->pabLineBuf, s->HWParams.iReversedHead);
    p->iBytesLeft = p->iBytesPerLine;
    s->iLinesLeft--;
  }

  /* copy (part of) a line */
  *len = MIN(maxlen, p->iBytesLeft);
  memcpy(buf, &p->pabLineBuf[p->iBytesPerLine - p->iBytesLeft], *len);
  p->iBytesLeft -= *len;

  DBG(DBG_MSG, " read=%d\n", *len);
#endif

  return SANE_STATUS_GOOD;
}


void
sane_cancel(SANE_Handle h)
{
  TScanner *s;

  DBG(DBG_MSG, "sane_cancel\n");

  s = (TScanner *)h;

  /* to be implemented more thoroughly */

  /* Make sure the scanner head returns home */
  FinishScan(&s->HWParams);

  s->fCanceled = TRUE;
  s->fScanning = FALSE;
}


SANE_Status
sane_set_io_mode(SANE_Handle h, SANE_Bool m)
{
  DBG(DBG_MSG, "sane_set_io_mode %s\n", m ? "non-blocking" : "blocking");

  /* prevent compiler from complaining about unused parameters */
  h = h;

  if (m) {
    return SANE_STATUS_UNSUPPORTED;
  }
  return SANE_STATUS_GOOD;
}


SANE_Status
sane_get_select_fd(SANE_Handle h, SANE_Int *fd)
{
  DBG(DBG_MSG, "sane_select_fd\n");

  /* prevent compiler from complaining about unused parameters */
  h = h;
  fd = fd;

  return SANE_STATUS_UNSUPPORTED;
}

