package BarCode;
use strict;

# Copyright Martijn van Oosterhout 2002. All Rights Reserved.
# This module may be distributed under the terms of the 
# GNU General Public License.

=head1 NAME

BarCode - Module to produce the barcodes required for cheap letters via Australia Post

=head1 SYNOPSIS

  use BarCode;

  my $bc = new BarCode;

  $bc->setType("11");    # Type 11 is the basic type, the 
                         # only type supported right now

  $bc->setDPID($dpid);   # Set the DPID to output

  print $bc->outputString,"\n";  # Display output

  # Outputs something like: 1301011030302221120200330212131030213
 
  # For embedding into PostScript
  print OUT "300 300 moveto\n";
  print OUT $bc->outputPS;

=head1 DESCRIPTION

  As of July 2002 you can no longer reduce your posting costs by presorting
  your letters. Instead, now you have to barcode them. This module takes the
  DPID which you got from somewhere else and calculates the string you need
  to output the barcode.

  Since I needed it in PostScript, there is a function for that. If you need
  another format, you may add it easily enough.

=head1 AUTHOR

  Martijn van Oosterhout Copyright 2002. All rights Reserved.

  This module may be distributed under the terms of the
  GNU General Public License.

=cut

use ReedSolomon;

# Table mapping digits for 4-state code
my @Ntable = qw(00 01 02 10 11 12 20 21 22 30);

# Takes a string of digits and converts it according to the N-code
sub ConvertTypeN ($)
{
  my $dpid = shift;
  die if $dpid =~ /[^0-9]/;

  $dpid =~ s/./$Ntable[$&]/ge;

  return $dpid;
}

# Make an instance of the class. Doesn't do much!!
sub new
{
  my $class = shift;

  my $self = bless {}, $class;

  return $self;
}

# Set the DPID
sub setDPID ($)
{
  my ($self,$dpid) = @_;

  $self->{dpid} = $dpid;
}

# Set the type. Type 11 is the only one supported right now.
sub setType ($)
{
  my ($self,$type) = @_;
    
  $self->{type} = $type;

}

# Take the given DPID and return the 4-state code string
sub outputString
{
  my $self = shift;
  my($type,$dpid,$info) = @$self{'type','dpid','info'};

  my $val;
  if( $type eq "11" )
  {
    die unless length($dpid) == 8;

    # Convert to the 21 digit string that represents the DPID
    $val = ConvertTypeN( $type.$dpid )."3";
  }
  else {die}  # This is where we would add support for other types

  # Breaks text into groups of three and converts them to numbers
  my @syms = map { my $v = 0; map { $v = $v*4 + $_ } split //; $v } grep { length } split /(...)/, $val;

  # Calculate the Reed-Solomon codes for this string
  my @codes = ReedSolomon::Encode( @syms );

  # Represent the codes as bars also
  my $code = join("", map { ($_ >> 4), (($_ >> 2) & 3), ($_ & 3) } @codes );

  # Add header and trailer
  my $out = "13".$val.$code."13";

  return $out;
}

# Instead of returning the strings, returns a snippet of PostScript to draw it
sub outputPS
{
  my $self = shift;

  # Get the string to draw
  my $str = $self->outputString;

  # Requirements:
  # Bar density: 22-25 bars per 25.4mm  =>  1.01-1.15mm/bar
  # Minimum gap: 0.4mm
  # Bar width: 0.4-0.6mm

  # In this code we make bar width 0.5mm and space 0.6mm
  my $barwidth = 0.5;
  my $barspace = 0.6;

  # Ascender height: 2.1-2.9mm
  # Tracker height: 0.5-0.8mm

  # In this code we make ascender height 2.5mm and tracker height 0.6mm
  my $maxheight = 2.5;
  my $minheight = 0.6;

  my $fullheight = $maxheight*2;
  my $halfheight = $maxheight+$minheight;
  my $midheight  = $maxheight-$minheight;
  my $tinyheight = $minheight*2;

  my $fullspace = $barspace + $barwidth;

  # Setup code and translate millimetres to points
  my $out = "gsave\n0 0 0 setrgbcolor\n2.83 2.83 scale\n";

  for my $char ( split //, $str )
  {
    # Small setup for each digit
    $out .= "gsave\nnewpath\n0 0 moveto\n";
    if( $char eq "0" )   # H 
    { $out .= "$barwidth 0 rlineto\n0 -$fullheight rlineto\n-$barwidth 0 rlineto\n0 $fullheight rlineto\n" }
    elsif( $char eq "1" )  # A (Ascender)
    { $out .= "$barwidth 0 rlineto\n0 -$halfheight rlineto\n-$barwidth 0 rlineto\n0 $halfheight rlineto\n" }
    elsif( $char eq "2" )  # D (Descender)
    { $out .= "0 -$midheight rmoveto\n$barwidth 0 rlineto\n0 -$halfheight rlineto\n-$barwidth 0 rlineto\n0 $halfheight rlineto\n" }
    elsif( $char eq "3" )  # T (Tracker)
    { $out .= "0 -$midheight rmoveto\n$barwidth 0 rlineto\n0 -$tinyheight rlineto\n-$barwidth 0 rlineto\n0 $tinyheight rlineto\n" }
    else { die }
    $out .= "fill\ngrestore\n$fullspace 0 translate\n";
  }

  # End snippet
  $out .= "grestore\n";

  return $out;
}

1;
