package ReedSolomon;
use strict;

# Copyright Martijn van Oosterhout 2002. All Rights Reserved.
# This module may be distributed under the terms of the 
# GNU General Public License.
# based on the C source code from Australia Post's website

# Module to calculate the Reed-Solomon codes for a string of integers as
# required for Australia Post barcodes. The algorithm is fairly well known
# and example code can be found on their website.

# This module pretty much does it the same way, though using perl certainly
# simplifies some of the manipulations....

=head1 NAME

ReedSolomon - Calculates Reed-Solomon codes for the purposes of Australia Post barcoding

=head1 SYNOPSIS

  Encode( @array );

=head1 DESCRIPTION

  Takes an array of integers in the range 0-63 and returns an array of four
  integers representing the Reed-Solomon codes.

  The algorithm is fairly well known and example code can be found on
  Australia Posts website.

  This module pretty much does it the same way, though using perl certainly
  simplifies some of the manipulations....

=head1 AUTHOR

  Martijn van Oosterhout Copyright 2002. All rights Reserved.

  This module may be distributed under the terms of the
  GNU General Public License.

=cut

my $numbits = 6;  # Number of bits (3 bars * 2bits/bar)
my $poly = 67;    # 1000011

my $maxval = 2**$numbits;  # 64 = limit of bits

my @multtable;   # Multiplication table
my @genpoly;

# Initialises the multtable
sub init
{
  for my $i (0..63)
  {
    $multtable[0][$i] = 0;
    $multtable[1][$i] = $i;
  }

  my $prev = 1;
  my $next;
  for my $i (1..63)
  {
    $next = $prev << 1;
    if( $next & $maxval )   # Over limit?
    { $next ^= $poly }

    for my $j (0..63)
    {
      $multtable[$next][$j] = $multtable[$prev][$j] << 1;

      if( $multtable[$next][$j] & $maxval )   # Over limit?
      { $multtable[$next][$j] ^= $poly }
    }
    $prev = $next;
  }

  # Sets the generator polynomial
  @genpoly = ( 48, 17, 29, 30, 1 );
}

sub Encode
{
  my @syms = reverse @_;

  if( not @genpoly ) { init() };

  my $k = scalar(@syms);
  my $n = $k + 4;

  my @temp = (0,0,0,0,@syms);

  for( my $i = $k-1; $i >= 0; $i-- )
  {
    for my $j (0..4)
    {
#      print STDERR "\$temp[ $i + $j ] ^= \$multtable[ $genpoly[$j] ][ $temp[ 4+$i ] ]\n";

      $temp[ $i + $j ] ^= $multtable[ $genpoly[$j] ][ $temp[ 4+$i ] ];
#      print STDERR "($i,$j,$temp[ $i + $j ]) ";
    }
  }
  print "\n",join(",",@temp),"\n";

  return reverse @temp[0..3];
}

1;
