/*
 * DNS utils for OpenGate
 * 
 * Copyright (c) Marco Polci
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * $Log: DNSUtils.cxx,v $
 * Revision 1.3  2000/11/24 09:53:17  mpolci
 * Changed the type of Environ::Dns from PString to PIPSocket::Address
 *
 * Revision 1.2  2000/10/17 13:00:13  mpolci
 * File header and CVS log entry setup
 *
 *
 * Revision 1.0  2000/08/23 16:06
 * Initial revision
 *
 */
#include <ptlib.h>
#include <ptlib/sockets.h>
#include <ptlib/svcproc.h>
#include "Environ.h"

#include "DNSUtils.h"

#define BUFFSIZE 4096       

#define DNSPORT 53
#define DNSMSGHDR_SIZE 12
#define DNSMSGMAXSIZE 256
#ifndef T_TXT
#define T_TXT 16
#endif
 
union DnsHdrFlags {
   struct {
      BYTE b2; // first octet
      BYTE b1; // second octet
   } byte;
   struct {
      unsigned
               rcode:4,
               z:3,
               ra:1,
               rd:1,
               tc:1,
               aa:1,
               opcode:4,
               qr:1;
   } field;
};

const BYTE dnsmsghdr_query[] = 
         {0xFB, 0x15, 0x01, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

unsigned prepare_query(BYTE *buff, const PString & domain)
  // Task: fill buff with a dns message to query for TEXT field of
  // indicated domain. Return the size of message.
{
   BYTE *cursor = buff;

   // prepare header
   memcpy(cursor, dnsmsghdr_query, DNSMSGHDR_SIZE);
   cursor += DNSMSGHDR_SIZE;
   
   // prepare question section
   // QNAME field (domain name)
   PINDEX old_pos = 0,
          new_pos;
   while ( (new_pos = domain.Find('.', old_pos)) != P_MAX_INDEX) {
      unsigned len = new_pos - old_pos;
      if (len < 64) {
         *cursor++ = (BYTE) len;
         memcpy(cursor, (const unsigned char *) domain(old_pos, new_pos - 1), len);
         cursor += len;
      } else {
         // domain name is not valid
         return 0;
      }
      old_pos = new_pos + 1;
   }
   if (old_pos < domain.GetLength() - 1) {
      new_pos = domain.GetLength();
      unsigned len = new_pos - old_pos;
      if (len < 64) {
         *cursor++ = (BYTE) len;
         memcpy(cursor, (const unsigned char *) domain(old_pos, new_pos - 1), len);
         cursor += len;
      } else {
         // domain name is not valid
         return 0;
      }
      *cursor++ = 0;
   }

   // QTYPE field (TEXT)
   *cursor++ = 0x00;
   *cursor++ = 0x10;

   // QCLASS field (INET)
   *cursor++ = 0x00;
   *cursor++ = 0x01;

   return cursor - buff;
}

PINDEX make_query(const Environ & Environ, const PString & domain, BYTE * resp)
  // Task: make a query to a DNS for TEXT fields of specified domain. Fill resp
  // whith reply and return the number of bytes reads.
{
   BYTE query[BUFFSIZE];
   unsigned len;
   PUDPSocket sock;
   
   if ( (len = prepare_query(query, domain)) != 0) {
      //if (!sock.WriteTo(query, len, PIPSocket::Address(Environ.Dns), DNSPORT))
      if (!sock.WriteTo(query, len, Environ.Dns, DNSPORT))
         return 0;
      PIPSocket::Address addr;
      WORD port;

      // delete next two lines
      // sock.GetLocalAddress(addr, port);
      // PSYSTEMLOG(Info, "make_query() : sock opened to " << addr << " " << port);

      sock.SetReadTimeout(3000);
      if (!sock.ReadFrom(resp, BUFFSIZE, addr, port))
         return 0;
   }
   return sock.GetLastReadCount();
}

BYTE *skip_name(BYTE *ptr, BYTE *msg, BYTE *msg_limit)
  // Task: get a response message of DNS and return a pointer after last byte of
  // name fields pointed by ptr. msg and msg_limit are boundaries of message.
{
   BYTE len;
   do {
      if (ptr >= msg_limit) // error
         return msg_limit;
      len = *ptr++;
      switch (len & 0xC0) {
         case 0x00:       // length OCTET
            ptr += len;
            break;
         case 0x40:       // undefined, treat it as a pointer
         case 0x80:
            PSYSTEMLOG(Error, "DNS msg: name field not valid");
         case 0xC0:       // pointer 
            ++ptr;
            len = 0; // exit from loop
            break;
      } 
   } while (len != 0);
   return ptr;
}

BOOL extract_text_response(BYTE *resp, unsigned msgsize, PStringList &strlst)
  // Task: get a response message of DNS, extract TEXT fields and puts they in
  // strlst. Return TRUE if no errors occurs.
{
   BYTE *cursor = resp;
   BYTE *resp_limit = resp + msgsize;
   WORD qdcount, ancount;
   DnsHdrFlags flags;
   
   // parse header section
   
   // skip id
   cursor += 2;
   // flags
   flags.byte.b1 = *cursor++;
   flags.byte.b2 = *cursor++;
   // qdcount
   qdcount = PSocket::Net2Host(*(WORD *)cursor);
   cursor += 2;
   // ancount
   ancount = PSocket::Net2Host(*(WORD *)cursor);
   cursor += 2;
   // skip nscout and arcount
   cursor += 4;

   // check flags. Exit if is not a response or had some error condition
   if ( flags.field.qr != 1 || flags.field.rcode != 0 ) 
      return FALSE;
      
   // skip questions section

   for (WORD i = 0; i < qdcount; i++) {
      // skip QNAME
      cursor = skip_name(cursor, resp, resp_limit);
      // skip QTYPE and QCLASS
      cursor += 4;
      if (cursor > resp_limit) return FALSE;
   }

   // parse answers section

   for (WORD i = 0; i < ancount; i++) {
      WORD a_type,
           a_class,
           a_rdlength;
      DWORD a_ttl;
      // skip NAME
      cursor = skip_name(cursor, resp, resp_limit);
      if (cursor >= resp_limit - 10) return FALSE;
      // TYPE
      a_type = PSocket::Net2Host(*(WORD *)cursor);
      cursor += 2;
      // CLASS
      a_class = PSocket::Net2Host(*(WORD *)cursor);
      cursor += 2;
      // TTL
      a_ttl = PSocket::Net2Host(*(DWORD *)cursor);
      cursor += 4;
      // RDLENGTH
      a_rdlength = PSocket::Net2Host(*(WORD *)cursor);
      cursor += 2;
      // RDATA
      if (a_type == T_TXT) {
         BYTE len = *cursor;
         PString str((char *) cursor+1, len);
         strlst.AppendString(str);
      }
      cursor += a_rdlength;
   }
   return TRUE;
}

BOOL skip_spaces(const PString & str, PINDEX & i)
{
   if (str[i] != ' ' && str[i] != '\t') return FALSE;
   ++i;
   while (str[i] == ' ' || str[i] == '\t') ++i;
   return TRUE;
}

BOOL GetGKNameFromTextField(const PString &text, PString &name)
  // Task: extract name of gatekeeper from a DNS TEXT string and 
  // set name to it. Return TRUE if successful.
{
   PString str = text.Trim();
   PINDEX i = 0;
   if ( !(str(i,i+2) *= PString("ras")) ) return FALSE;
   i += 3;
   if (!skip_spaces(str, i)) return FALSE;
   PINDEX next_tok = str.Find('@', i);
   if (next_tok != P_MAX_INDEX)
      i = next_tok + 1;
   next_tok = str.FindOneOf("\t :", i);
   if (next_tok == P_MAX_INDEX) next_tok = str.GetLength();
   name = str(i, next_tok - 1);
   return TRUE;
}

int DNSUtils::QueryDNSForGK(const Environ & Environ, const PString & domain, AddressList & gks)
  // Task: to query DNS for gatekeepers relative to the domain indicated. Puts
  //       addresses in gks and returns the number of gatekeepers found.
{
   //PSYSTEMLOG(Info, "DNSUtils::QueryDNSForGK");
   BYTE buff[BUFFSIZE];
   PINDEX size;

   size = make_query(Environ, domain, buff);
   if (size > 0) {
      PStringList texts;
      if (extract_text_response(buff, size, texts)) {
         int nAddr = 0;
         for (PINDEX i = 0; i < texts.GetSize(); i++) {
            PString gk;
            if (GetGKNameFromTextField(texts[i], gk)) {
               PIPSocket::Address addrGK;
               if (PIPSocket::GetHostAddress(gk, addrGK)) {
                  gks.push_back(addrGK);
                  ++nAddr;
               } else {
                  PString msg = "Gatekeeper address is not valid - " + gk;
                  PSYSTEMLOG(Error, msg);
               }
            }
         }
         return nAddr;
      }
   }
   PSYSTEMLOG(Error, "Cannot query dns");
   return 0;
}


/*

   PPipeChannel pc(Environ.QueryDNSCmd + " " + domain, PPipeChannel::ReadOnly);
   pc.WaitForTermination();
   if (pc.Read(buff, BUFFSIZE)) {
      PStringArray gknames = PString(buff, pc.GetLastReadCount()).Lines();
      int nAddr = 0;
      for (unsigned i = 0; i < gknames.GetSize(); ++i) {
         //PSYSTEMLOG(Info, gknames[i]);
         PIPSocket::Address addrGK;
         if (PIPSocket::GetHostAddress(gknames[i], addrGK)) {
            gks.push_back(addrGK);
            ++nAddr;
         } else {
            PString msg = "Gatekeeper address is not valid - " + gknames[i];
            PSYSTEMLOG(Error, msg);
         }
      }
      return nAddr;
   } else {
      //int err = pc.GetErrorCode();
      PSYSTEMLOG(Error, "Cannot query dns");
      return 0;
   }

}

*/
