/*
 * Endpoint Table for OpenGate
 * 
 * Copyright (c) Egoboo Ltd. 1999-2000
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Open Gatekeeper
 *
 * The Initial Developer of the Original Code is Egoboo Ltd.
 *
 * $Log: EndpointTabl.cxx,v $
 * Revision 1.21  2000/05/25 14:25:30  aunitt
 * Fixed bug that stopped gateway prefixes working.
 *
 * Revision 1.20  2000/05/23 10:17:00  aunitt
 * Added support for statically defined gateway prefixes.
 *
 * Revision 1.19  2000/05/15 19:06:01  aunitt
 * Changed GetRasIPAddress to return full address structure not just IP address
 * structure.
 *
 * Revision 1.18  2000/05/12 14:08:50  aunitt
 * Made sure we only do bogus includes when necessary.
 *
 * Revision 1.17  2000/05/12 10:20:33  aunitt
 * Changed the way that searches for gateway prefixes are handled by creating
 * a seperate list of gateways instead of doing a search of all endpoints.
 * Merged removal code into a single function.
 *
 * Revision 1.16  2000/05/05 11:23:35  aunitt
 * Added bandwidth management support.
 * Improved implementation of function to get time to live for an endpoint.
 *
 * Revision 1.15  2000/05/02 10:25:10  aunitt
 * Added support for setting endpoint time to live from parameter in RRQ.
 *
 * Revision 1.14  2000/04/27 18:05:49  aunitt
 * Completed changes necessary to support registration timeouts.
 *
 * Revision 1.13  2000/04/26 17:11:32  aunitt
 * Added initial support for registration time to live.
 *
 * Revision 1.12  2000/04/21 13:44:59  aunitt
 * Removed bogus warnings in Microsoft headers.
 *
 * Revision 1.11  2000/04/20 18:53:19  aunitt
 * Added support for storing vendor information.
 * Added support for storing information on endpoints that are not registered with
 * us but are discovered by other means (e.g. LRQ to neighbour)
 *
 * Revision 1.10  2000/04/17 17:40:45  aunitt
 * Changed underlying structure for the Endpoint table from a list to a set
 * (ordering by endpoint id)
 * Added a map for aliases to endpoint id to speed up lookups.
 *
 * Revision 1.9  2000/04/13 18:46:38  aunitt
 * Added function to remove bogus addresses (e.g. 0.0.0.0) from an endpoint before
 * adding it to the table.
 *
 * Revision 1.8  2000/04/12 13:32:01  aunitt
 * Fixed Windows compile problem.
 *
 * Revision 1.7  2000/04/10 19:20:00  aunitt
 * Made search for protocol descriptor check for Voice and not H323.
 *
 * Revision 1.6  2000/04/05 15:22:30  aunitt
 * Added support for reading supported prefixes from the capabilities description
 * supplied by a Gatekeeper.
 *
 * Revision 1.5  2000/03/24 18:57:24  aunitt
 * Fixed bug in doing RRJs that caused null pointer accesses.
 *
 * Revision 1.4  2000/03/21 21:17:54  aunitt
 * Added function to check if an endpoint has all the given aliases in an array.
 *
 * Revision 1.3  2000/02/27 20:48:20  aunitt
 * Added windows support.
 *
 * Revision 1.2  2000/02/15 20:45:54  aunitt
 * Added support for E164 prefixes for Gateways
 *
 * Revision 1.1.1.1  2000/02/01 22:25:34  aunitt
 * Initial revision
 *
 *
 */

#if (_MSC_VER >= 1200)  
#pragma warning( disable : 4800 ) // remove performance warning about bool...
#pragma warning( disable : 4710 ) // remove warning about inlining
#endif

#include <ptlib.h>
#include <ptlib/svcproc.h>
#if (_MSC_VER >= 1200)
#include <ptlib/sockets.h>  // Needed to stop ambigious symbol error
#include <q931.h>           // Ditto
#include <h245.h>
#pragma warning( push, 3 )
#endif
#include <algorithm>
#if (_MSC_VER >= 1200)  
#pragma warning( pop )
#endif

#include "AddrUtils.h"
#include "Environ.h"
#include "EndpointTabl.h"


Endpoint::Endpoint( const H225_ArrayOf_TransportAddress & RasAddr,
                    const H225_ArrayOf_TransportAddress & CallSigAddr,
		            const H225_EndpointType &             AkaType,
                    const H225_ArrayOf_AliasAddress &     AkaAlias,
                    const H225_EndpointIdentifier &       AkaId,
                    const H225_VendorIdentifier &         AkaVendor,
                          bool                            AkaIsLocal
		  ) : RasAddress(RasAddr), CallSignalAddr(CallSigAddr),
		      Alias(AkaAlias), Id(AkaId), TypeKnown(true), VendorKnown(true),
		      IsLocal(AkaIsLocal), Type(AkaType), Vendor(AkaVendor)
{
}

Endpoint::Endpoint( const H225_ArrayOf_TransportAddress & RasAddr,
                    const H225_ArrayOf_TransportAddress & CallSigAddr,
		            const H225_EndpointType &             AkaType,
                    const H225_ArrayOf_AliasAddress &     AkaAlias,
                    const H225_EndpointIdentifier &       AkaId,
                          bool                            AkaIsLocal
		  ) : RasAddress(RasAddr), CallSignalAddr(CallSigAddr),
		      Alias(AkaAlias), Id(AkaId), TypeKnown(true), VendorKnown(false),
		      IsLocal(AkaIsLocal), Type(AkaType)
{
}

Endpoint::Endpoint( const H225_ArrayOf_TransportAddress & RasAddr,
                    const H225_ArrayOf_TransportAddress & CallSigAddr,
                    const H225_ArrayOf_AliasAddress &     AkaAlias,
                    const H225_EndpointIdentifier &       AkaId,
                          bool                            AkaIsLocal
		  ) : RasAddress(RasAddr), CallSignalAddr(CallSigAddr),
		      Alias(AkaAlias), Id(AkaId), TypeKnown(false), VendorKnown(false),
		      IsLocal(AkaIsLocal)
{
}

Endpoint::Endpoint( const H225_ArrayOf_TransportAddress & CallSigAddr,
                    const H225_ArrayOf_AliasAddress &     AkaAlias,
                    const H225_EndpointIdentifier &	      AkaId
                  ) : CallSignalAddr(CallSigAddr), Alias(AkaAlias), Id(AkaId),
                      TypeKnown(false), VendorKnown(false), IsLocal(false)
{
}

Endpoint::Endpoint( const H225_ArrayOf_AliasAddress &     AkaAlias,
                    const H225_EndpointIdentifier &	      AkaId
                  ) : Alias(AkaAlias), Id(AkaId),
                      TypeKnown(false), VendorKnown(false), IsLocal(false)
{
}

Endpoint::Endpoint( const H225_EndpointIdentifier & AkaId ) : Id(AkaId)
{
}

Endpoint::~Endpoint()
{
	// Nothing to do?
}

bool Endpoint::HasRasAddr( const H225_TransportAddress & Addr ) const
// Task: returns true iff this Endpoint has this RAS address
{
	for ( PINDEX i=0; i < RasAddress.GetSize(); ++i )
	{
		if ( RasAddress[i] == Addr )
			return true;
	}

	return false;
}

bool Endpoint::HasCallAddr( const H225_TransportAddress & Addr ) const
// Task: returns true iff this Endpoint has this call signalling address
{
	for ( PINDEX i=0; i < CallSignalAddr.GetSize(); ++i )
	{
		if ( CallSignalAddr[i] == Addr )
			return true;
	}

	return false;
}

bool Endpoint::IsGateway() const
{
    return ( TypeKnown && Type.HasOptionalField( H225_EndpointType::e_gateway ) );
}

static bool IsPrefix( const H225_AliasAddress & Prefix,
                      const H225_AliasAddress & Alias
                    )
// Task: to return true iff the given prefix is a prefix of the given alias
{
    // Check they are both the same type
    if ( Prefix.GetTag() == Alias.GetTag() )
    {
        PString PrefixStr;
        PString AliasStr;

        // Now check by type
        switch ( Prefix.GetTag() )
        {
            case H225_AliasAddress::e_e164          :
                PrefixStr = static_cast<const PASN_IA5String &>(Prefix);
                AliasStr  = static_cast<const PASN_IA5String &>(Alias);
                break;
            case H225_AliasAddress::e_partyNumber   :
                {
                    const H225_PartyNumber & PrefixPartyNumber = Prefix;
                    const H225_PartyNumber & AliasPartyNumber  = Alias;
                    if ( PrefixPartyNumber.GetTag() != AliasPartyNumber.GetTag() )
                    {
                        // Not the same type of party number
                        return false;
                    }
                    switch( PrefixPartyNumber.GetTag() )
                    {
                        case H225_PartyNumber::e_publicNumber               :
                        {
                            const H225_PublicPartyNumber & PublicPrefix = PrefixPartyNumber;
                            const H225_PublicPartyNumber & PublicAlias  = AliasPartyNumber;
                            if ( PublicPrefix.m_publicTypeOfNumber != PublicAlias.m_publicTypeOfNumber )
                            {
                                return false;
                            }
                            PrefixStr = static_cast<const PASN_IA5String &>(PublicPrefix.m_publicNumberDigits);
                            AliasStr  = static_cast<const PASN_IA5String &>(PublicAlias.m_publicNumberDigits);
                            break;
                        }
                        case H225_PartyNumber::e_privateNumber              :
                        {
                            const H225_PrivatePartyNumber & PrivatePrefix = PrefixPartyNumber;
                            const H225_PrivatePartyNumber & PrivateAlias  = AliasPartyNumber;
                            if ( PrivatePrefix.m_privateTypeOfNumber != PrivateAlias.m_privateTypeOfNumber )
                            {
                                return false;
                            }
                            PrefixStr = static_cast<const PASN_IA5String &>(PrivatePrefix.m_privateNumberDigits);
                            AliasStr  = static_cast<const PASN_IA5String &>(PrivateAlias.m_privateNumberDigits);
                            break;
                        }
                        case H225_PartyNumber::e_dataPartyNumber            :
                        case H225_PartyNumber::e_telexPartyNumber           :
                        case H225_PartyNumber::e_nationalStandardPartyNumber:
                        default                                             :
                            PrefixStr = static_cast<const PASN_IA5String &>(PrefixPartyNumber);
                            AliasStr  = static_cast<const PASN_IA5String &>(AliasPartyNumber);
                            break;
                    }
                }
                break;
            default                                 :
                PrefixStr = static_cast<const PASN_BMPString &>(Prefix);
                AliasStr  = static_cast<const PASN_BMPString &>(Alias);
                break;
        }

        return ( strncmp( PrefixStr, AliasStr, PrefixStr.GetLength() ) == 0 );
    }

    return false;
}

static bool IsSupportedProtocolPrefix( const H225_VoiceCaps &       Protocol,
                                       const H225_AliasAddress &    Alias
                                     )
// Task; to return true iff the given alias is supported as a prefix of the given
//       gateway protocol descriptor
{
    if ( Protocol.HasOptionalField( H225_VoiceCaps::e_supportedPrefixes ) )
    {
        for ( PINDEX i=0; i < Protocol.m_supportedPrefixes.GetSize(); ++i )
        {
            if ( IsPrefix( Protocol.m_supportedPrefixes[i].m_prefix, Alias ) )
                return true;
        }
    }

    return false;
}

static bool AkaIsSupportedPrefix( const H225_GatewayInfo &     Info,
                                  const H225_AliasAddress &    Alias
                                )
// Task: to return true iff the given Alias matches a supported prefix of the given
//       gateway
{
    // First find the H323 protocol prefixes
    if ( Info.HasOptionalField( H225_GatewayInfo::e_protocol ) )
    {
        for ( PINDEX p=0; p < Info.m_protocol.GetSize(); ++p )
        {
            if ( Info.m_protocol[p].GetTag() == H225_SupportedProtocols::e_voice )
            {
                // Okay, found the voice protocol description
                // Is this correct?? What protocols should we be checking?
                H225_VoiceCaps & Caps = Info.m_protocol[p];

                if ( IsSupportedProtocolPrefix( Caps, Alias ) )
                    return true;
            }
        }
    }

    return false;
}

bool Endpoint::IsSupportedPrefix( const H225_AliasAddress & AkaAlias ) const
// Task: to return true iff this alias is a supported prefix on this endpoint.
//       Assumes this endpoint is a gateway.
{
    PAssert( IsGateway(), "Trying to check for a prefix but I'm not a gateway" );
    	
    // If this endpoint is a gateway it should provide a list of supported
    // prefixes in the capabilities section for each protocol it supports

    // Check supported prefixes in Gateway Info
    if ( AkaIsSupportedPrefix( Type.m_gateway, AkaAlias ) )	
        return true;
	
	// Check E164 prefixes - is this needed or is it just plain wrong?
	// Leave it in here for the moment
    for ( PINDEX i=0; i < Alias.GetSize(); ++i )
    {
        if ( IsPrefix( Alias[i], AkaAlias ) )
            return true;
    }

    return false;
}

bool Endpoint::HasAlias( const H225_AliasAddress & AkaAlias ) const
// Task: returns true iff this Endpoint has this alias
{
	for ( PINDEX i=0; i < Alias.GetSize(); ++i )
	{
		if ( Alias[i] == AkaAlias )
			return true;	
	}
	
	if ( IsGateway() )
	    return IsSupportedPrefix( AkaAlias );

	return false;
}

bool Endpoint::HasAllAliases( const H225_ArrayOf_AliasAddress & Aliases ) const
// Task: to return true iff we have have the given aliases
{
	for ( PINDEX i=0; i < Aliases.GetSize(); ++i )
	{
	    if ( !HasAlias(Aliases[i]) )
	        return false;
	}
	
	return true;
}

bool Endpoint::operator==( const Endpoint & EP ) const
{
	if ( this == &EP )
		return true;

	return ( ( RasAddress     == EP.RasAddress     ) &&
		     ( CallSignalAddr == EP.CallSignalAddr ) &&
	         ( Alias          == EP.Alias          ) &&
             ( Type           == EP.Type           ) &&
             ( Id             == EP.Id             )
	       );
}

bool Endpoint::operator!=( const Endpoint & EP ) const
{
	return ( ( RasAddress     != EP.RasAddress     ) ||
		     ( CallSignalAddr != EP.CallSignalAddr ) ||
	         ( Alias          != EP.Alias          ) ||
             ( Type           != EP.Type           ) ||
             ( Id             != EP.Id             )
	       );
}

bool Endpoint::operator<( const Endpoint & EP ) const
{
	return ( Id < EP.Id );
}

bool Endpoint::GetRasIPAddress( H225_TransportAddress & Addr ) const
// Task: to return the Ras IP address for this endpoint
//       (the endpoint can have many different types of address, but
//       but we normally use the IP address)
{
    try
    {
        AddrUtils::GetIPAddress( Addr, RasAddress );
        return true;
    }
    catch( AddrUtils::IPAddrNotFoundError Err )
    {
        return false;
    }
}

bool Endpoint::GetCallSigIPAddress( H225_TransportAddress_ipAddress & Addr ) const
// Task: to return the call signalling IP address for this endpoint
{
    try
    {
        AddrUtils::GetIPAddress( Addr, CallSignalAddr );
        return true;
    }
    catch( AddrUtils::IPAddrNotFoundError Err )
    {
        return false;
    }
}

class TableCleaner : public PTimer
{
    // It is the responsibility of the table cleaner to unregister any endpoints
    // that are pending unregistration due to expired time to live
    // Note: we are relying on the fact that only one timeout can be active at a time
    PCLASSINFO( TableCleaner, PTimer );
    public:
        TableCleaner( EndpointTable * EPTable );
        virtual ~TableCleaner();

        virtual void OnTimeout();
        // Task: to handle the timeout condition of the cleaner, i.e. remove endpoints
        //       scheduled for removal

        void ScheduleUnregistration( const H225_EndpointIdentifier & Id );
        // Task: to schedule the given Endpoint for unregistration

    protected:
        TableCleaner();                         // Prevent call to default constructor
        TableCleaner(const TableCleaner &TC);   // and copy constructor

        PLIST(IdList,H225_EndpointIdentifier);
        EndpointTable * EPTable;
        IdList          ToBeUnregistered;
        PMutex          Mutex;
};

TableCleaner::TableCleaner( EndpointTable * Table ) : PTimer(0), EPTable(Table)
{
}

TableCleaner::~TableCleaner()
{
}

void TableCleaner::ScheduleUnregistration( const H225_EndpointIdentifier & Id )
// Task: to schedule the given Endpoint for unregistration
{
    Mutex.Wait();
    ToBeUnregistered.Append(Id.Clone());
    Mutex.Signal();

    // Cause us to be triggered shortly
    PTimeInterval time(1);
    PTimer *Timer = this;
    *Timer = time;
}

void TableCleaner::OnTimeout()
// Task: to do the work of the table cleaner....
{
    while( !ToBeUnregistered.IsEmpty() )
    {
        Mutex.Wait();
        H225_EndpointIdentifier Id = ToBeUnregistered[0];
        ToBeUnregistered.RemoveAt(0);
        Mutex.Signal();
        try
        {
            EPTable->Remove( EPTable->FindByEndpointId( Id ) );
        }
        catch( EndpointTable::NotFoundError )
        {
            PSYSTEMLOG( Warning, "Couldn't find id " << Id << " when cleaning table" );
        }
    }
}


static const PTimeInterval GracePeriod( 0, 3 );
// Grace time to allow for network delays and other unforseen things to give
// the endpoint time to say it's still alive

class TimeToLiveEntry : public PTimer
{
    PCLASSINFO( TimeToLiveEntry, PTimer );
    public:
        TimeToLiveEntry( const H225_EndpointIdentifier &    Id,
                         const PTimeInterval &              TTL,
                               RegistrationTimeoutHdlr *    Hdlr
                       );
        // Use this constructor for creating a TTL entry

        TimeToLiveEntry( const H225_EndpointIdentifier & Id );
        // Degenerate constructor for use in comparisons

        virtual ~TimeToLiveEntry();

        virtual void OnTimeout();
        // Task: Handle a timeout of the TTL for an Endpoint

        virtual Comparison Compare(const PObject & obj) const;
        // PWLIB comparison function

        void RestartTimer( const PTimeInterval & TTL );
        // Task: to restart the timer

        PTimeInterval GetTTL() const;
        // Task: to return the time to live of this entry
   		
    protected:
        TimeToLiveEntry();      // Disallow calls to default constructor
        TimeToLiveEntry(const TimeToLiveEntry &E);

        H225_EndpointIdentifier     Id;
        RegistrationTimeoutHdlr *   Hdlr;
        unsigned                    Chance;
};

TimeToLiveEntry::TimeToLiveEntry( const H225_EndpointIdentifier &   AkaId,
                                  const PTimeInterval &             TTL,
                                        RegistrationTimeoutHdlr *   AkaHdlr
                                ) : PTimer( TTL + GracePeriod ), Id( AkaId ),
                                    Hdlr( AkaHdlr ), Chance( 0 )
{
}

TimeToLiveEntry::TimeToLiveEntry( const H225_EndpointIdentifier & AkaId ) : Id( AkaId )
{
    Hdlr = NULL;
}

TimeToLiveEntry::~TimeToLiveEntry()
{
}

void TimeToLiveEntry::OnTimeout()
// Task: Handle a timeout of the TTL for an Endpoint
{
    PTimeInterval NextTimeout;

    if ( ( Hdlr != NULL ) && Hdlr->OnRegistrationTimeout( Id, Chance++, NextTimeout ) )
    {
        PTimer *Timer = this;
        *Timer = NextTimeout;
    }
}

PObject::Comparison TimeToLiveEntry::Compare(const PObject & obj) const
// Comparison function
{
    // NOTE: The assumption is that we only ever have a maximum of one timeout per id
    // existing at any one time.
    PAssert(obj.IsDescendant(TimeToLiveEntry::Class()), PInvalidCast);
    const TimeToLiveEntry & other = (const TimeToLiveEntry &)obj;
    // Compare using endpoint ids....
    return Id.Compare( other.Id );
}

void TimeToLiveEntry::RestartTimer( const PTimeInterval & TTL )
// Task: to restart the timer
{
    Chance = 0;
    PTimer *Timer = this;
    *Timer = TTL + GracePeriod;
}

PTimeInterval TimeToLiveEntry::GetTTL() const
// Task: to return the time to live of this entry
{
    // This is probably not terrible accurate as I think this value will only get
    // updated when a timer expires or a new timer is added.
    // This is fine for our purposes now, but in the future we may need to force
    // a timer update here.

    // Note - we lie a little in that we take account of the grace period....
    const PTimeInterval *Time = this;
    if ( *Time > GracePeriod )
        return (*Time - GracePeriod);
    else
        return 0;
}



struct hasAlias
{
	const H225_AliasAddress & Alias;
	hasAlias(const H225_AliasAddress &AkaAlias) : Alias(AkaAlias) {}
	bool operator()( const Endpoint & EP ) const
	{
		return EP.HasAlias( Alias );
	}
};

struct hasAnyAlias
{
	const H225_ArrayOf_AliasAddress & Alias;
	hasAnyAlias(const H225_ArrayOf_AliasAddress &AkaAlias) : Alias(AkaAlias) {}
	bool operator()( const Endpoint & EP ) const
	{
		for ( PINDEX i = 0; i < Alias.GetSize(); ++i )
			if ( EP.HasAlias(Alias[i]) )
				return true;
		return false;
	}
};

struct hasRasAddr
{
	const H225_TransportAddress & Addr;
	hasRasAddr(const H225_TransportAddress &AkaAddr) : Addr(AkaAddr) {}
	bool operator()( const Endpoint & EP ) const
	{
		return EP.HasRasAddr( Addr );
	}
};

struct hasAnyRasAddr
{
	const H225_ArrayOf_TransportAddress & Addr;
	hasAnyRasAddr(const H225_ArrayOf_TransportAddress &AkaAddr) : Addr(AkaAddr) {}
	bool operator()( const Endpoint & EP ) const
	{
		for ( PINDEX i = 0; i < Addr.GetSize(); ++i )
			if ( EP.HasRasAddr(Addr[i]) )
				return true;
		return false;
	}
};

struct hasCallAddr
{
	const H225_TransportAddress & Addr;
	hasCallAddr(const H225_TransportAddress &AkaAddr) : Addr(AkaAddr) {}
	bool operator()( const Endpoint & EP ) const
	{
		return EP.HasCallAddr( Addr );
	}
};

struct hasAnyCallAddr
{
	const H225_ArrayOf_TransportAddress & Addr;
	hasAnyCallAddr(const H225_ArrayOf_TransportAddress &AkaAddr) : Addr(AkaAddr) {}
	bool operator()( const Endpoint & EP ) const
	{
		for ( PINDEX i = 0; i < Addr.GetSize(); ++i )
			if ( EP.HasCallAddr(Addr[i]) )
				return true;
		return false;
	}
};

struct hasEndpointId
{
	const H225_EndpointIdentifier & Id;
	hasEndpointId(const H225_EndpointIdentifier &AkaId) : Id(AkaId) {}
	bool operator()( const Endpoint & EP ) const
	{
		return (EP.GetId() == Id);
	}
};


EndpointTable::EndpointTable( const Environ & Env ) :
    MyEnviron(Env), GatekeeperId(MyEnviron.MyId)
{
    Cleaner     = new TableCleaner(this);
    TimeoutHdlr = NULL;
}

EndpointTable::~EndpointTable()
{
    delete Cleaner;
}

H225_EndpointIdentifier EndpointTable::GenerateEndpointId() const
// Task: to generate a new, unique Endpoint Id
{
	static unsigned         Id=0;
	H225_EndpointIdentifier Result;

	// Try generating a new id and check that it isn't already in the table
	// (because the counter can wrap round)
	do
	{
		((PASN_BMPString &)Result) = PString(PString::Unsigned, Id) + ": " + (PString)GatekeeperId;
		++Id;
	} while ( find_if(Table.begin(),Table.end(),hasEndpointId(Result)) != Table.end() );

	return Result;
}

static void GenerateAlias( const H225_TransportAddress &     Addr,
                                 H225_ArrayOf_AliasAddress & Alias 
                         )
// Task: to generate a new unique alias, uses the transport address as this should
//       be unique
{
	Alias.SetSize(1);
	Alias[0].SetTag( H225_AliasAddress::e_transportID );
	(H225_TransportAddress &)Alias[0] = Addr;
}

static void GenerateAlias( const H225_EndpointIdentifier &      Id,
                                 H225_ArrayOf_AliasAddress &    Alias
                         )
// Task: alternate generator using endpoint id....
{
   	Alias.SetSize(1);
   	Alias[0].SetTag( H225_AliasAddress::e_h323_ID );
   	(PASN_BMPString &)Alias[0] = (PASN_BMPString &)Id;
}

static void GetDuplicateAliases(       EndpointTable *             Table, 
                                 const H225_ArrayOf_AliasAddress & Aliases, 
                                       H225_ArrayOf_AliasAddress & Duplicates
                               )
// Task: to search the given table for the given aliases and return all those that
//       are found in duplicates
{
	PINDEX j = 0;
	Duplicates.SetSize(0);

	for ( PINDEX i=0; i < Aliases.GetSize(); ++i )
	{
		if ( Table->AliasExists( Aliases[i] ) )
		{
			Duplicates.SetSize( j+1 );
			Duplicates[j] = Aliases[i];
			++j;
		}
	}
}

static bool IsInvalidTransportAddr( const H225_TransportAddress & Addr )
// Task: to return true iff the given transport address is invalid
{
    if ( Addr.GetTag() == H225_TransportAddress::e_ipAddress )
    {
        // Check against 0.0.0.0
        const H225_TransportAddress_ipAddress & AddrIP = Addr;
        return ( ( AddrIP.m_ip[0] == 0 ) && ( AddrIP.m_ip[1] == 0 ) &&
                 ( AddrIP.m_ip[2] == 0 ) && ( AddrIP.m_ip[3] == 0 )
               );
    }

    return false;
}

static void RemoveBogusAddrs( H225_ArrayOf_TransportAddress & Addrs )
// Task: to remove any invalid transport and call address from the given array of
//       transport addresses
{
    PINDEX i=0;

    while ( i < Addrs.GetSize() )
    {
        if ( IsInvalidTransportAddr( Addrs[i] ) )
            Addrs.RemoveAt(i);
        else
            ++i;
    }
}

static void RemoveBogusAddrs( Endpoint & EP )
// Task: to remove any invalid transport and call address from the given endpoint.
{
    RemoveBogusAddrs( EP.GetWriteableRasAddresses() );
    RemoveBogusAddrs( EP.GetWriteableCallSigAddresses() );
}

static void InsertAliases( EndpointTable::AliasMap_t &          Map,
                           const H225_ArrayOf_AliasAddress &    Aliases,
                           const H225_EndpointIdentifier &      Id
                         )
// Task: to insert into the given map mappings from the given aliases to the given
//       endpoint id.
// Note: should be called when mutual exclusion to the table has been obtained
{
    for ( PINDEX i=0; i < Aliases.GetSize(); ++i )
	{
	    Map[Aliases[i]] = Id;
	}
}

void EndpointTable::Insert( Endpoint &                      EP,
                            const H225_TransportAddress &   RegAddr,
                            PTimeInterval                   EP_TTL
                          )
// Task: insert the given endpoint into the table with the given time to live
//       Will throw an error if the insert fails because constraints
//       are violated (e.g. alias is already registered with a different
//       transport address)
//       Will update the values of an endpoint if it already exists but
//       the aliases have changed
//       Will assign an alias to the endpoint if it hasn't got one
{ 	
    // Remove bogus addresses (e.g. 0.0.0.0) from supplied addresses
    RemoveBogusAddrs( EP );

	// Get the aliases this endpoint is trying to register
	const H225_ArrayOf_AliasAddress & Aliases       = EP.GetAliases();
	const H225_ArrayOf_TransportAddress & CallAddrs = EP.GetCallSigAddresses();
	const H225_ArrayOf_TransportAddress & RasAddrs  = EP.GetRasAddresses();
	
	// Basic sanity checks
	if ( EP.IsInLocalZone() )
	{
    	if ( CallAddrs.GetSize() < 1 )
	    {
		    throw RegistrationError( H225_RegistrationRejectReason::e_invalidCallSignalAddress,
		                             "Call signalling address not defined"
		                           );
    	}

	    if ( RasAddrs.GetSize() < 1 )
	    {
		    throw RegistrationError( H225_RegistrationRejectReason::e_invalidRASAddress,
		                             "RAS address not defined"
		                           );
    	}
    }

	// See if the endpoint has an alias, if not generate one
	if ( Aliases.GetSize() == 0 )
	{
	    // Try and generate an alias using a transport address if possible, otherwise
	    // fall back to using the endpoint id which is always guaranteed to be available
		H225_ArrayOf_AliasAddress NewAlias;
		if ( RasAddrs.GetSize() > 0 )
    		GenerateAlias( RasAddrs[0], NewAlias );
    	else if ( CallAddrs.GetSize() > 0 )
    	    GenerateAlias( CallAddrs[0], NewAlias );
    	else
    	    GenerateAlias( EP.GetId(), NewAlias );
		EP.UpdateAlias( NewAlias );
	}	

	// Similarly for call signalling and RAS addresses
	if ( CallAddrs.GetSize() > 1 )
	{
		TableMutex.Wait();
	    size_t Count = count_if(Table.begin(), Table.end(), hasAnyCallAddr(CallAddrs));
    	TableMutex.Signal();
	    if ( Count > 1 )
	    {
		    throw RegistrationError( H225_RegistrationRejectReason::e_invalidCallSignalAddress,
		                             "Call signalling addresses already belong to multiple endpoints"
		                           );
    	}
	}
	
	if ( RasAddrs.GetSize() > 1 )
	{
		TableMutex.Wait();
    	size_t Count = count_if(Table.begin(), Table.end(), hasAnyRasAddr(RasAddrs));
   		TableMutex.Signal();
	    if ( Count > 1 )
    	{
	    	throw RegistrationError( H225_RegistrationRejectReason::e_invalidRASAddress,
		                             "RAS addresses already belong to multiple endpoints"
		                           );
    	}
	}

	// See if the alias already exists in the table and it's ours....
	try
	{
		Endpoint ExistingEP = FindByAlias( Aliases );
		if ( ( ExistingEP.GetRasAddresses()     != RasAddrs  ) ||  // Assume != operator is okay?
		     ( ExistingEP.GetCallSigAddresses() != CallAddrs )
		   )
		{
			H225_RegistrationRejectReason Reason;
			Reason.SetTag( H225_RegistrationRejectReason::e_duplicateAlias );
			H225_ArrayOf_AliasAddress &   Duplicates = Reason;

			GetDuplicateAliases( this, Aliases, Duplicates );
			
			throw RegistrationError( Reason,
			                         "Alias already registered to different endpoint"
			                       );
		}

		// Remove stale entry
		Remove( ExistingEP );
	}
	catch ( EndpointTable::InconsistentError )
	{
		// Aliases are already registered to multiple endpoints
		H225_RegistrationRejectReason Reason;
		Reason.SetTag( H225_RegistrationRejectReason::e_duplicateAlias );
		H225_ArrayOf_AliasAddress &   Duplicates = Reason;

		GetDuplicateAliases( this, Aliases, Duplicates );

		throw RegistrationError( Reason,
 		                         "Alias already assigned to multiple endpoints"
				       );
	}
	catch ( EndpointTable::NotFoundError )
	{
		// Alias not registered, check if the transport address is,
		// if it is we should replace it with the new registration

		RemoveByRasAddr( RasAddrs );
		RemoveByCallSigAddr( CallAddrs );
    }

    // Create registration TTL entry
   	TimeToLiveEntry * TimeToLive = new TimeToLiveEntry( EP.GetId(),
   	                                                    min( EP_TTL,
   	                                                         MyEnviron.GetEndpointTTL()
   	                                                       ),
   	                                                    TimeoutHdlr
   	                                                  );
	
	TableMutex.Wait();
	Table.insert( EP );
	InsertAliases( AliasMap, Aliases, EP.GetId() );
	if ( EP.IsInLocalZone() && EP.IsGateway() )
	{
	    // Add any additional static prefixes
	    PIPSocket::Address  Addr;
	    WORD                Port;
	    H225_EndpointType   Type = EP.GetType();
	
	    AddrUtils::ConvertToIPAddress( RegAddr, Addr, Port );
	    MyEnviron.AddStaticPrefixes( Type.m_gateway, Addr );
	    EP.SetType( Type );
	
	    // Add to the list of gateways
	    Gateways.Append( EP.GetId().Clone() );
	}
	TTL.Append( TimeToLive );
	TableMutex.Signal();
}

static void RemoveAliases( EndpointTable::AliasMap_t &          Map,
                           const H225_ArrayOf_AliasAddress &    Aliases
                         )
// Task: to remove from the given map mappings for the given aliases
// Note: should be called when mutual exclusion to the table has been obtained
{
    for ( PINDEX i=0; i < Aliases.GetSize(); ++i )
	{
       	EndpointTable::AliasMap_t::iterator iter = Map.find( Aliases[i] );
       	if ( iter != Map.end() )
       	{
       	    Map.erase(iter);
        }	
	}
}

static void RemoveRegistrationTTLEntry( TTLList &                       TTL,
                                        const H225_EndpointIdentifier & Id
                                      )
// Task: to remove the given endpoint from the registration TTL set
// Note: should be called when mutual exclusion to the table has been obtained
{
    TimeToLiveEntry Target( Id );
    PINDEX i = TTL.GetValuesIndex( Target );
    if ( i != P_MAX_INDEX )
    {
        TTL.RemoveAt(i);
    }
}

static void RemoveGateway( EndpointTable::GatewayList_t &   Gateways,
                           const H225_EndpointIdentifier &  Id
                         )
// Task: to remove the given id from the list of gateways
// Note: should be called when mutual exclusion to the table has been obtained
{
    PINDEX i = Gateways.GetValuesIndex( Id );
    if ( i != P_MAX_INDEX )
    {
        Gateways.RemoveAt(i);
    }
}

void EndpointTable::DoRemove( Table_t::iterator i )
// Task: to do the actual removal of the given entry in the table.
//       Assumes mutual exclusion has been obtained.
{
	if ( i != Table.end() )
	{
        RemoveRegistrationTTLEntry( TTL, (*i).GetId() );
	    RemoveAliases( AliasMap, (*i).GetAliases() );
	    if ( (*i).IsGateway() )
	        RemoveGateway( Gateways, (*i).GetId() );
	    Table.erase(i);
	}
}

void EndpointTable::Remove( const Endpoint & EP )
// Task: to remove the given endpoint from the table
{
	TableMutex.Wait();
	Table_t::iterator i = Table.find(EP);
    DoRemove(i);
	TableMutex.Signal();
}

void EndpointTable::RemoveByRasAddr( const H225_ArrayOf_TransportAddress & Addr )
// Task: to remove the endpoint with the given RAS addresses
{
    try
    {
        Remove( FindByRasAddr( Addr ) );
    }
    catch( EndpointTable::NotFoundError )
    {
        // Don't care - it's gone anyway....
    }
}

void EndpointTable::RemoveByCallSigAddr( const H225_ArrayOf_TransportAddress & Addr )
// Task: to remove the endpoint with the given call signal addresses
{
    try
    {
        Remove( FindByCallSigAddr( Addr ) );
    }
    catch( EndpointTable::NotFoundError )
    {
        // Don't care - it's gone anyway....
    }
}

void EndpointTable::DeferredRemove( const H225_EndpointIdentifier & Id ) const
// Task: to remove the endpoint with the specified identifier as soon as possible.
//       Call this if you want to remove an endpoint during processing a timeout
{
    Cleaner->ScheduleUnregistration( Id );
}

bool EndpointTable::AliasExists( const H225_AliasAddress & Alias ) 
{
	TableMutex.Wait();
	AliasMap_t::const_iterator i = AliasMap.find( Alias );
	TableMutex.Signal();
	
	return i != AliasMap.end();
}

Endpoint EndpointTable::FindByPrefix( const H225_ArrayOf_AliasAddress & Alias )
// Task: to return the endpoint (gateway) that supports routing to a telephone number
//       prefix of the given alias, e.g. gateway that supports calling "020" when the
//       alias is "02073620359".
{
    PWaitAndSignal MutexWait(TableMutex);
    for ( PINDEX i=0; i < Alias.GetSize(); ++i )
    {
        if ( ( Alias[i].GetTag() == H225_AliasAddress::e_e164         ) ||
             ( Alias[i].GetTag() == H225_AliasAddress::e_partyNumber  )
           )
        {
            for ( PINDEX j=0; j < Gateways.GetSize(); ++j )
            {
                // We need to do this search ourselves rather than call
                // FindByEndpointId as we can't release the mutex on the table
                // otherwise the list of gateways could possibly get inconsistent
                Endpoint EP(Gateways[j]);
            	Table_t::const_iterator iter = Table.find(EP);
                if ( ( iter != Table.end() )                   &&
                     ( (*iter).IsSupportedPrefix( Alias[i] ) )
                   )
                {
                    return *iter;
                }
            }
        }
    }

    throw NotFoundError("Alias not found");
}

Endpoint EndpointTable::FindByAlias( const H225_ArrayOf_AliasAddress & Alias ) 
// Task: to return the endpoint that has the given alias, will throw
//       an exception if the alias doesn't exist or is defined to multiple
//       endpoints
{
	if ( Alias.GetSize() == 0 )
		throw NotFoundError("No alias specified");

	if ( Alias.GetSize() > 1 )
	{
		// Check consistency - i.e. do the requested aliases all refer to the
		// same endpoint
		TableMutex.Wait();
		bool Inconsistent = ( count_if(Table.begin(), Table.end(), hasAnyAlias(Alias)) > 1 );
		TableMutex.Signal();
		if ( Inconsistent )
			throw InconsistentError( InconsistentError::Alias, "Aliases inconsistent" );
	}
	
	TableMutex.Wait();
	// We can search just using the first alias as we know they are consistent....
	AliasMap_t::const_iterator i = AliasMap.find( Alias[0] );
	TableMutex.Signal();
	
	if ( i == AliasMap.end() )
	{
	    // Direct match for alias not found, try searching by prefix
	    return FindByPrefix( Alias );
	}

	// Use the endpoint Id returned by the map to find the endpoint....
	return FindByEndpointId((*i).second);
}

bool EndpointTable::RasAddrExists( const H225_TransportAddress & Addr ) 
// Task: to return true iff the given RAS address is already registered in
//       the table
{
	TableMutex.Wait();
	Table_t::const_iterator i = find_if(Table.begin(),
                                        Table.end(),
	                                    hasRasAddr(Addr)
	                                   );
	TableMutex.Signal();
	
	return i != Table.end();
}

Endpoint EndpointTable::FindByRasAddr( const H225_ArrayOf_TransportAddress & Addr ) 
// Task: to return the endpoint that has the given RAS address, will throw
//       an exception if the alias doesn't exist
{
	if ( Addr.GetSize() == 0 )
		throw NotFoundError("No RAS address specified");

	if ( Addr.GetSize() > 1 )
	{
		// Check consistency - i.e. do the requested addresses all refer to the
		// same endpoint
		TableMutex.Wait();
		bool Inconsistent = ( count_if(Table.begin(), Table.end(), hasAnyRasAddr(Addr)) > 1 );
		TableMutex.Signal();
		if ( Inconsistent )
			throw InconsistentError( InconsistentError::RASAddress, 
			                         "RAS addresses inconsistent"
			                       );
	}

	TableMutex.Wait();
    Table_t::const_iterator i = find_if(Table.begin(),
	                		            Table.end(),
	                                    hasRasAddr(Addr[0])
	                                   );
	TableMutex.Signal();
	
	if ( i == Table.end() )
		throw NotFoundError("RAS address not found");	

	return *i;
}

bool EndpointTable::CallSigAddrExists( const H225_TransportAddress & Addr ) 
// Task: to return true iff the given call signal address is already 
//       registered in the table
{
	TableMutex.Wait();
	Table_t::const_iterator i = find_if(Table.begin(),
	               		                Table.end(),
	                                    hasCallAddr(Addr)
	                                   );
	TableMutex.Signal();
	
	return i != Table.end();
}

Endpoint EndpointTable::FindByCallSigAddr( const H225_ArrayOf_TransportAddress & Addr ) 
// Task: to return the endpoint that has the given call signal address, 
//       will throw an exception if the alias doesn't exist
{
	if ( Addr.GetSize() == 0 )
		throw NotFoundError("No call signalling address specified");

	if ( Addr.GetSize() > 1 )
	{
		// Check consistency - i.e. do the requested addresses all refer to the
		// same endpoint
		TableMutex.Wait();
		bool Inconsistent = ( count_if(Table.begin(), Table.end(), hasAnyCallAddr(Addr)) > 1 );
		TableMutex.Signal();
		if ( Inconsistent )
			throw InconsistentError( InconsistentError::CallSignallingAddress, 
			                         "Call signalling addresses inconsistent"
			                       );
	}

	TableMutex.Wait();
	Table_t::const_iterator i = find_if(Table.begin(),
	                                    Table.end(),
	                                    hasCallAddr(Addr[0])
	                                   );
	TableMutex.Signal();
	
	if ( i == Table.end() )
		throw NotFoundError("Call signalling address not found");	

	return *i;
}

Endpoint EndpointTable::FindByEndpointId( const H225_EndpointIdentifier & Id ) 
// Task: to return the endpoint that has the given identifier
//	 will throw an exception if the endpoint id doesn't exist
{
    Endpoint EP(Id);

	TableMutex.Wait();
	Table_t::const_iterator i = Table.find(EP);
	TableMutex.Signal();	

	if ( i == Table.end() )
		throw NotFoundError("Endpoint Identifier not found");	

	return *i;
}

Endpoint EndpointTable::Pop()
// Task: pop the first endpoint from the table
{
	if ( Table.empty() )
		throw NotFoundError("Table is empty, cannot pop");

	TableMutex.Wait();
	Table_t::iterator i = Table.begin();
	Endpoint Result = *i;
	DoRemove(i);
	TableMutex.Signal();
	
	return Result;
}

void EndpointTable::ResetTTL( const H225_EndpointIdentifier & Id )
// Task: to reset the time to live of the given endpoint, i.e. let it live
//       longer
{
    TimeToLiveEntry Target( Id );
   	TableMutex.Wait();
    PINDEX i = TTL.GetValuesIndex( Target );
    if ( i == P_MAX_INDEX )
        PSYSTEMLOG( Warning, "Failed to find endpoint to update TTL :" << Id ) ;
    else
        // Restart the timer...
        TTL[i].RestartTimer( MyEnviron.GetEndpointTTL() );
    TableMutex.Signal();
}


H225_TimeToLive EndpointTable::GetTimeToLive( const H225_EndpointIdentifier & Id )
// Task: to return the time to live for the given endpoint
{
    H225_TimeToLive Result;
    TimeToLiveEntry Target( Id );
    PINDEX i = TTL.GetValuesIndex( Target );
    if ( i != P_MAX_INDEX )
    {
        Result.SetValue( TTL[i].GetTTL().GetSeconds() );
    }
    else
    {
        Result.SetValue( 0 );
    }

    return Result;
}


bool operator==( const H225_AliasAddress & a,
                 const H225_AliasAddress & b
               )
// Equality operator for alias addresses
{
	if ( a.GetTag() != b.GetTag() )
		return false;

	return ( a.Compare(b) == PObject::EqualTo );
}

bool operator==( const H225_TransportAddress & a,
                 const H225_TransportAddress & b
               )
// Equality operator for transport addresses
{
	if ( a.GetTag() != b.GetTag() )
		return false;

	return ( a.Compare(b) == PObject::EqualTo );
}

bool operator< ( const H225_EndpointIdentifier & a,
                 const H225_EndpointIdentifier & b
               )
// Less than operator for endpoint ids....
{
	return ( a.Compare( b ) == PObject::LessThan );
}

bool operator==( const H225_EndpointIdentifier & a,
                 const H225_EndpointIdentifier & b
               )
// Equality operator for endpoint ids....
{
	return ( a.Compare( b ) == PObject::EqualTo );
}

