/*
 * OpenGatekeeper environment definition
 *
 * Copyright (c) Egoboo Ltd. 1999-2000
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Open Gatekeeper
 *
 * The Initial Developer of the Original Code is Egoboo Ltd.
 *
 * $Log: Environ.h,v $
 * Revision 1.9  2000/05/23 10:13:54  aunitt
 * Added support for statically defined gateway prefixes.
 *
 * Revision 1.8  2000/05/12 14:07:30  aunitt
 * Renamed PRasLog to Log.
 * Made sure we only use includes when necessary.
 *
 * Revision 1.7  2000/05/10 09:14:36  aunitt
 * Added setting for routing H.245
 * Removed truncated identifier warning in Windows debug build.
 *
 * Revision 1.6  2000/05/05 11:20:24  aunitt
 * Added bandwidth management.
 *
 * Revision 1.5  2000/04/27 17:59:53  aunitt
 * Changed to use real value for endpoint TTL, no longer debugging.
 *
 * Revision 1.4  2000/04/26 17:09:44  aunitt
 * Added initial support for registration time to live.
 *
 * Revision 1.3  2000/04/21 13:42:43  aunitt
 * Removed bogus warnings.
 * Thanks Microsoft!
 *
 * Revision 1.2  2000/04/20 18:49:42  aunitt
 * Added function to read list of neighbouring Gatekeepers.
 *
 * Revision 1.1  2000/04/10 19:23:11  aunitt
 * Environment object for OpenGatekeeper.
 *
 *
 */

#ifndef _ENVIRON_H
#define _ENVIRON_H

#include <ptlib.h>
#include <ptlib/sockets.h>
#include <ptlib/timeint.h>
#include <h225.h>
#if (_MSC_VER >= 1200)
#include <q931.h>
#include <h245.h>
#pragma warning( disable : 4786 ) // Disable about debug information being truncated
#pragma warning( disable : 4800 ) // remove performance warning about bool...
#pragma warning( push, 2 )
#endif
#include <vector>
#if (_MSC_VER >= 1200)  
#pragma warning( pop )
#endif
#include "EndpointTabl.h"
#include "CallTabl.h"
#include "Log.h"
#include "ProxyCriteria.h"
#include "RouteTbl.h"


typedef vector<PIPSocket::Address> AddressList;

class Environ
{
    public:
        Environ()           { Cfg.SetDefaultSection("System"); };
        virtual ~Environ()  {};

        bool IsGatekeeperRouted() const { return Cfg.GetBoolean( "IsGKRouted" ); }
        //Task: to return true iff we should be using gatekeeper routed calls

        bool RouteH245() const { return Cfg.GetBoolean( "Route H245" ); }
        // Task: to return true iff we should also route H.245

        AddressList GetNeighbours() const;
        //Task: to return a list of the IP addresses of other gatekeepers we know about

        void AddStaticPrefixes( H225_GatewayInfo &          Info,
                                const PIPSocket::Address &  Addr
                              ) const;
        // Task: to add any statically defined prefixes for the gateway at the given address...

        PTimeInterval GetEndpointTTL() const;
        // Task: to return the time to live for an endpoint

        unsigned MaxBandwidth() const
        // Task: to return the maximum bandwidth to be used, 0 for unlimited bandwidth
        { return static_cast<unsigned>(Cfg.GetInteger( "Max Bandwidth", 0 )); }

        unsigned MinBandwidth() const
        // Task: to return the minimum bandwidth allocated per call, work around for
        //       NetMeeting bug(?)
        { return static_cast<unsigned>(Cfg.GetInteger( "Min Call Bandwidth", 10000 )); }

        bool CheckPortNumber() const { return Cfg.GetBoolean( "Check Port Number", true ); }
        // Task: to return true if the proxy should use only non system port numbers

        bool CheckIP() const { return Cfg.GetBoolean( "Check IP", true ); }
        // Task: to return true if the proxy should check the ip of the logical channels

        bool DisableARJToRouteCall() const { return Cfg.GetBoolean( "Disable ARJ To Route Call", false ); }
        // Task: to return true if the GK shouldn't use ARJ message with "route call to gatekeeper"
        //       motivation. This option disallow the routing of call signals when a registered endpoint
        //       reply to a call from a not registered endpoint.

        bool AcceptUnregisteredCalls() const { return Cfg.GetBoolean( "Accept Unregistered Calls", false ); }
        // Task: to return true if the GK should accept calls that aren't registered previously with an ARQ
        //       message. This imply that the GK reply to the LRQ messages with own call signal address
        //       instead of the endpoint's call signall address.

        const H225_TransportAddress & GetMyCallAddr() const { return MyCallAddr; };
        H225_TransportAddress         GetMyCallAddr(PIPSocket::Address destination) const;
        H225_TransportAddress         GetMyCallAddr(const H225_TransportAddress & destination) const;
        // Task: to retrieve the calling address of the gatekeeper. The first return the address on the default
        //       route interface. The others two returns the address on the intercace used to route the
        //       specified address.

        EndpointTable *             EPTable;
        CallTable *                 CTable;
        H225_GatekeeperIdentifier   MyId;
        H225_TransportAddress       MyCallAddr;
        PIPSocket::Address          LocalAddr;
        OpengateLog *               Log;
        PIPSocket::Address          Dns;
        ProxyCriteria               PrxCriteria;
        RouteTbl                    RouteTable;

    protected:
        H225_ArrayOf_SupportedPrefix GetStaticPrefixes( const PIPSocket::Address & Addr ) const;
        // Task: to return the array of additional prefixes defined for the given gateway
        //       in the list of static prefixes

        PConfig Cfg;
};

#endif // _ENVIRON_H
