/*
 * Proxy for OpenGate
 * 
 * Copyright (c) Marco Polci
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * $Log: Proxy.cxx,v $
 * Revision 1.4  2000/10/25 09:51:32  mpolci
 * Removed unused variable warning in metod ProxyUDPLCListener::Listening()
 *
 * Revision 1.3  2000/10/19 14:53:16  mpolci
 * Now UDPProxyLChannel::do_proxy throw the exception SocketError on read and write error
 *
 * Revision 1.2  2000/10/17 13:00:13  mpolci
 * File header and CVS log entry setup
 *
 *
 * Revision 1.0  2000/09/04 10:33
 * Initial revision
 *
 */

#include <ptlib.h>
#include <ptlib/svcproc.h>
#include <ptlib/sockets.h>
#include "Proxy.h"

#define STACKSIZE 8192
#define BUFFLEN 2048

#define CATCH_ALL(msg, INSTR) \
                      catch (runtime_error & e) { PSYSTEMLOG( Error, msg << ": " << e.what() ); INSTR; } \
                      catch (logic_error & e) { PSYSTEMLOG( Error, msg << ": " << e.what() ); INSTR;} \
                      catch ( ... ) { PSYSTEMLOG( Error, msg << ": unknow error" ); INSTR; } 

inline BOOL odd(WORD n)
{
   return n & 1;
}


UDPProxyLChannel::UDPProxyLChannel(Type d) : ProxyLChannel(d)
{
}

UDPProxyLChannel::~UDPProxyLChannel()
{
}

void UDPProxyLChannel::bindSockets()
{
   s_src.Listen();
   s_dst.Listen();
}

void UDPProxyLChannel::setSource(const PIPSocket::Address & addr, WORD port)
{
   ProxyLChannel::setSource(addr, port);
   //PSYSTEMLOG(Info, "UDPSocket to " << a_src << " port " << src_port);
   s_src.SetSendAddress(a_src, src_port);
}

void UDPProxyLChannel::setDestination(const PIPSocket::Address & addr, WORD port)
{
   ProxyLChannel::setDestination(addr, port);
   //PSYSTEMLOG(Info, "UDPSocket to " << a_dst << " port " << dst_port);
   s_dst.SetSendAddress(a_dst, dst_port);
}

void UDPProxyLChannel::getSourceSideAddr(PIPSocket::Address & addr, WORD & port)
{ 
   s_src.GetLocalAddress(addr, port);  
   //PSYSTEMLOG(Info, "GetLocalAddress() = " << addr << " port " << port);
}

void UDPProxyLChannel::getDestinationSideAddr(PIPSocket::Address & addr, WORD & port)
{ 
   s_dst.GetLocalAddress(addr, port);  
   //PSYSTEMLOG(Info, "GetLocalAddress() = " << addr << " port " << port);
}

void UDPProxyLChannel::open()
{
   // Must check if sockets have some problems and throw an exception

   fOpened = TRUE;
}

ProxyLCListener * UDPProxyLChannel::allocateListener(PThread::AutoDeleteFlag deletion)
{
   return new ProxyUDPLCListener(*this, deletion);
}

ProxyLChannel::Direction UDPProxyLChannel::proxyMsg(const PTimeInterval & ti_timeout)
// Task: Waits for data on both source and destination sockets, reads it
// and forward it in the other socket. The case of unidirectional channel
// is treated as the case of bidirectional channel.
{
   if (!fOpened) throw runtime_error("Channel not opened");

   int selres = PSocket::Select(s_src, s_dst, ti_timeout);   
   
//   PSYSTEMLOG(Info, "proxyMsg() - PSocket::Select() = " << selres);

   switch (selres) {
      case -1: // src socket
         do_proxy(s_src, s_dst);
         return forward;
      case -2: // dst socket
         do_proxy(s_dst, s_src);
         return reverse;
      case -3: // both sockets
         do_bidirectional_proxy();
         return both;
      case 0: // timeout
         return timeout;
         break;
      default: // some error?
         // throw some exception ?
         break;
   }
   return timeout; // to avoid warning. Never reaches this line.
}

void UDPProxyLChannel::do_proxy(PUDPSocket & rd, PUDPSocket & wr)
// Task: reads from rd socket and forward the data writing it on wr socket
{
   PIPSocket::Address addr;
   WORD port;
   PINDEX len;
   BYTE *buff = new BYTE[BUFFLEN];

   //PSYSTEMLOG( Info, "UDPProxyLChannel::do_proxy()");

   if ( rd.ReadFrom(buff, BUFFLEN, addr, port) &&
        (len = rd.GetLastReadCount()) > 0
      ) 
   {
      //PSYSTEMLOG( Info, "UDPProxyLChannel::do_proxy() letti " << len << " byte - buff = " << (void *) buff);
      if (len == BUFFLEN) PSYSTEMLOG(Info, "Reading buffer filled");

      // must check if addr and port are the same of source
      if (!wr.Write(buff, len)) {
         // must throw an exception
         PIPSocket::Address laddr, saddr, sladdr;
         WORD lport, sport, slport;
         rd.GetLocalAddress(laddr, lport);
         wr.GetSendAddress(saddr, sport);
         wr.GetLocalAddress(sladdr, slport);

         PSYSTEMLOG( Error, "proxyMsg: udp write error on port " << slport << " destination " 
                     << saddr << ":" << sport << " (data read from " << addr << ":"
                     << port << " on port " << lport << ")" 
                   );
         throw SocketError("write error");
      }
   } else {
      PSYSTEMLOG(Error, "proxyMsg: read error");
      throw SocketError("read error");
   }

   delete buff;
}

void UDPProxyLChannel::do_bidirectional_proxy()
// Task: reads from both src and dst sockets and forward the messages
// writing they respectly on dst and src socket
{
   BYTE *buff_s, *buff_d;
   PIPSocket::Address addr_s, addr_d;
   WORD port_s, port_d;
   PINDEX len_s, len_d;
   BOOL f_s, f_d;

   PSYSTEMLOG(Info, "UDPProxyLChannel::do_bidirectional_proxy()");

   buff_s = new BYTE[BUFFLEN];
   buff_d = new BYTE[BUFFLEN];

   f_s = s_src.ReadFrom(buff_s, BUFFLEN, addr_s, port_s);
   f_d = s_dst.ReadFrom(buff_d, BUFFLEN, addr_d, port_d);

   if ( f_s && 
        (len_s = s_src.GetLastReadCount()) > 0
      ) 
   {
      if (len_s == BUFFLEN) PSYSTEMLOG(Info, "Reading buffer filled");

      // must check if addr and port are the same of source
      if (!s_dst.Write(buff_s, len_s)) {
         // must throw an exception
         PSYSTEMLOG(Error, "proxyMsg: udp write error (data read from " << addr_s << " : " << port_s << ")");
      }
   } else {
      // must throw an exception
      PSYSTEMLOG(Error, "proxyMsg: read error");
   }

   if ( f_d && 
        (len_d = s_dst.GetLastReadCount()) > 0
      ) 
   {
      if (len_d == BUFFLEN) PSYSTEMLOG(Info, "Reading buffer filled");

      // must check if addr and port are the same of source
      if (!s_src.Write(buff_d, len_d)) {
         // must throw an exception
         PSYSTEMLOG(Error, "proxyMsg: udp write error (data read from " << addr_d << " : " << port_d << ")");
      }
   } else {
      // must throw an exception
      PSYSTEMLOG(Error, "proxyMsg: read error");
   }

   delete buff_s;
   delete buff_d;
}


/************************************************************************************************/

RTPProxyLChannel::RTPProxyLChannel(Type d) : UDPProxyLChannel(d), fDeleteRTCP(TRUE)
{
   rtcp = new RTCPProxyLChannel(this, d);
}

RTPProxyLChannel::~RTPProxyLChannel()
{
  if (fDeleteRTCP) {
     delete rtcp;
  }
}

void RTPProxyLChannel::bindSockets()
{
  PSYSTEMLOG(Info, "RTPProxyLChannel::bindSockets()");
  if (rtcp == NULL)
     PSYSTEMLOG(Error, "rtcp cannot be null!");
  bind(s_src, rtcp->s_src);
  bind(s_dst, rtcp->s_dst);
} 

void RTPProxyLChannel::bind(PUDPSocket & rtp, PUDPSocket & rtcp)
// Task: bind rtp to an even port and rtcp to the next one port
{
   WORD port_rtp;
   bool fOk = false;
   
   do {
      while ( !rtp.Listen(5, port_rtp) ) {
         if (port_rtp == 0)
            throw BindError("Cannot bind udp sockets for rtp");
         else {
            // try next port number
            rtp.Close();
            ++port_rtp;
            //PSYSTEMLOG( Info, "try next port");
         }
      }
      port_rtp = rtp.GetPort();
      //PSYSTEMLOG( Info, "loop - port = " << port_rtp );
      if ( odd(port_rtp) ) {
         // rtp port must be even
         rtp.Close();
         ++port_rtp;
         //PSYSTEMLOG( Info, "continue");
         continue;
      }
      //PSYSTEMLOG( Info, "try rctp" );
      // rtcp port must be next one the rtp port 
      if ( rtcp.Listen(5, port_rtp + 1) )
         fOk = true;
      else {
         //PSYSTEMLOG( Info, "failed rctp" );
         rtp.Close();
         rtcp.Close();
      }

   } while (!fOk);
   //PSYSTEMLOG( Info, "fine loop - ports = " << port_rtp << " " << port_rtp+1);
}

// void RTPProxyLChannel::bind(PUDPSocket & rtp, PUDPSocket & rtcp)
// // Task: bind rtp to an even port and rtcp to the next one port
// {
//    WORD port_rtp;
//    bool fOk = false;
//    do {
//       if ( !rtp.Listen(5, 0) )
//          throw BindError("Cannot bind udp sockets for rtp");
//       port_rtp = rtp.GetPort();
//       PSYSTEMLOG( Info, "loop - port = " << port_rtp );
//       if ( odd(port_rtp) ) {
//          // rtp port must be even
//          rtp.Close();
//          if ( !rtp.Listen(5, ++port_rtp) ) {
//             rtp.Close();
//             PSYSTEMLOG( Info, "continue");
//             continue;
//          }
//       }
//       PSYSTEMLOG( Info, "try rctp" );
//       // rtcp port must be next one the rtp port 
//       if ( rtcp.Listen(5, port_rtp + 1) )
//          fOk = true;
//       else {
//          PSYSTEMLOG( Info, "failed rctp" );
//          rtp.Close();
//          rtcp.Close();
//       }
//    } while (!fOk);
//    PSYSTEMLOG( Info, "fine loop - ports = " << port_rtp << " " << port_rtp+1);
// }

RTCPProxyLChannel::RTCPProxyLChannel(RTPProxyLChannel * p_rtp, Type d) : UDPProxyLChannel(d), rtp(p_rtp)
{
   if (rtp == NULL) 
      throw invalid_argument("Null point as rtp proxy channel");
}

RTCPProxyLChannel::~RTCPProxyLChannel()
{
}

void RTCPProxyLChannel::bindSockets()
// Nothing to to. The corresponding RTPProxyLChannel object binds the sockets
{
}

/********************************************************************/

TCPProxyLChannel::TCPProxyLChannel(Type d) : ProxyLChannel(d)
{
  //PSYSTEMLOG( Info, "TCPProxyLChannel::TCPProxyLChannel()" );
}

TCPProxyLChannel::~TCPProxyLChannel()
{
   s_listener.Close();
}

void TCPProxyLChannel::bindSockets()
{
  if ( !s_listener.Listen() ) {
     PSYSTEMLOG(Error, "Cannot bind listener socket for TCPProxyLChannel");
     throw BindError("Cannot bind listener socket for TCPProxyLChannel");
  }
}

void TCPProxyLChannel::getSourceSideAddr(PIPSocket::Address & addr, WORD & port)
{ 
   s_listener.GetLocalAddress(addr, port);
   //PSYSTEMLOG(Info, " TCPProxyLChannel::getSourceSideAddr() : " << addr << " port " << port);
}

ProxyLCListener * TCPProxyLChannel::allocateListener(PThread::AutoDeleteFlag deletion)
{
   return new ProxyTCPLCListener(*this, deletion);
}

//void TCPProxyLChannel::open()
TCPProxyLChannel::TCPConnectedChannel * TCPProxyLChannel::acceptCall()
{
   PIPSocket::Address addr;
   WORD port;
   s_listener.GetLocalAddress(addr, port);
   //PSYSTEMLOG( Info, "TCPProxyLChannel::accepCall() Waiting for call on port " << port << " ... " );

   TCPConnectedChannel *pcc = new TCPConnectedChannel;

   if ( !pcc->s_src.Accept(s_listener) )
   {
      PSYSTEMLOG(Error, "TCPProxyLChannel::accepCall() Cannot accept connection");
      delete pcc;
      throw ProxyLChannel::OpenError("Cannot accept connection");
   }

   pcc->s_dst.SetPort(dst_port);
   if ( !pcc->s_dst.Connect(a_dst) ) 
   {
      PSYSTEMLOG(Error, "TCPProxyLChannel::acceptCall() Cannot connect to " << a_dst << " port " << dst_port);
      delete pcc;
      throw ProxyLChannel::OpenError("Cannot open TCPProxyLChhannel");
   }
   //PSYSTEMLOG(Info, "TCPProxyLChannel::acceptCall() Estabilished TCP connession to " << a_dst 
   //           << " port " << dst_port);

   fOpened = TRUE;

   return pcc;
}

ProxyLChannel::Direction TCPProxyLChannel::TCPConnectedChannel::proxyMsg(const PTimeInterval & ti_timeout)
// Task: Waits for data on both source and destination sockets, reads it
// and forward it in the other socket. The case of unidirectional channel
// is treated as the case of bidirectional channel.
{
   int selres = PSocket::Select(s_src, s_dst, ti_timeout);   
   
//   PSYSTEMLOG(Info, "proxyMsg() - PSocket::Select() = " << selres);

   switch (selres) {
      case -1: // src socket
         do_proxy(s_src, s_dst);
         return forward;
      case -2: // dst socket
         do_proxy(s_dst, s_src);
         return reverse;
      case -3: // both sockets
         do_proxy(s_src, s_dst);
         do_proxy(s_dst, s_src);
         return both;
      case 0: // timeout
         return timeout;
         break;
      default: // some error?
         // throw some exception ?
         break;
   }
   return timeout; // to avoid warning. Never reaches this line.
}

void TCPProxyLChannel::TCPConnectedChannel::do_proxy(PTCPSocket & rd, PTCPSocket & wr)
// Task: reads from rd socket and forward the data writing it on wr socket
{
   BYTE *buff = new BYTE[BUFFLEN];

   if ( rd.Read(buff, BUFFLEN) ) 
   {
      PINDEX len = rd.GetLastReadCount();

      /*
      PIPSocket::Address la_r, pa_r, la_w, pa_w;
      WORD lp_r, pp_r, lp_w, pp_w;
      rd.GetLocalAddress(la_r, lp_r);
      rd.GetPeerAddress(pa_r, pp_r);
      wr.GetLocalAddress(la_w, lp_w);
      wr.GetPeerAddress(pa_w, pp_w);
      PSYSTEMLOG(Info, "Read " << len << " bytes from " << pa_r << " : " << pp_r << " on port " << lp_r 
                       << " Sending to " << pa_w << " : " << pp_w << " through port " << lp_w);
      */

      if ( len > 0 ) 
      {
         if (len == BUFFLEN) PSYSTEMLOG(Info, "Reading buffer filled");

         if (!wr.Write(buff, len)) {
           PSYSTEMLOG(Error, "proxyMsg: tcp write error");
           throw ConnectionClose("Connection to called endpoint lost");
         }
      }
   } else {
      throw ConnectionClose("Connection to caller endpoint lost");
   }

   delete buff;
}

/********************************************************************/

ProxyLCListener::ProxyLCListener(ProxyLChannel &ch, AutoDeleteFlag deletion) : 
   PThread(STACKSIZE, deletion, HighPriority), plc(ch), fStart(FALSE)
{
   // To make sure that the thread start suspended, resume now and suspend
   // as first instruction in Main()
   Resume();
}

ProxyLCListener::~ProxyLCListener()
{
}

void ProxyLCListener::Main()
{
   //PSYSTEMLOG(Info, "ProxyLCListener::Main() - wait for startListening ...");
   while (!fStart)
      Suspend();
   PSYSTEMLOG(Info, "Starting listening ...");
   try {
      Listening();
   } 
   catch ( runtime_error & e ) {
      PSYSTEMLOG(Error, e.what());
   } 
   catch ( ... ) {
      PSYSTEMLOG( Error, "Unknow error" );
   }
   PSYSTEMLOG(Info, "... terminated listening");
}

void ProxyLCListener::startListening()
{ 
   fStart = TRUE; 
   Resume(); 
}

void ProxyLCListener::terminateListening()
{
   fStop = TRUE;
   if (!fStart) startListening();
}

//////////////////////////////////////////////////////////////

ProxyUDPLCListener::ProxyUDPLCListener(UDPProxyLChannel & ch, AutoDeleteFlag deletion)
  : ProxyLCListener(ch, deletion)
{
}

void ProxyUDPLCListener::Listening()
{
   channel().open();
   fStop = FALSE;
   while (!fStop) {
      // ProxyLChannel::Direction d =
            channel().proxyMsg(5000);

      //if (d == ProxyLChannel::reverse || d == ProxyLChannel::both) 
        //PSYSTEMLOG( Info, "data in reverse direction" );
   } 
}

//////////////////////////////////////////////////////////////

const PTimeInterval ProxyTCPLCListener::LISTENINGTIMEOUT = 5000;
const PTimeInterval ProxyTCPLCListener::DEATHTIMEOUT     = 7000;   // must be greater than LISTENINGTIMEOUT

ProxyTCPLCListener::ProxyTCPLCListener(TCPProxyLChannel & ch, AutoDeleteFlag deletion)
  : ProxyLCListener(ch, deletion)
{
}

void ProxyTCPLCListener::Listening()
{
   //PSYSTEMLOG(Info, "ProxyTCPLCListener::Listening()");
   channel().open();
   fStop = FALSE;
   while (!fStop) {
      try {
         TCPProxyLChannel::TCPConnectedChannel * cc = channel().acceptCall();
         ChannelListener * cclistener = new ChannelListener(this, cc);
         cclistener->Resume();
      }
      catch(ProxyLChannel::OpenError & e) {
         fStop = TRUE;
      }
      CATCH_ALL("Cannot accept tcp call for proxy operations", (fStop = TRUE) );
   }
   Sleep(DEATHTIMEOUT);
}

ProxyTCPLCListener::ChannelListener::ChannelListener( ProxyTCPLCListener                    * pThrParent,
                                                      TCPProxyLChannel::TCPConnectedChannel * s,
                                                      AutoDeleteFlag                          deletion
                                                    )
  : PThread(STACKSIZE, deletion /*, HighPriority*/), parent(pThrParent), psockets(s)
{
   if (s == NULL) 
      throw invalid_argument("Null pointer as tcp connected channel");
   if (pThrParent == NULL)
      throw invalid_argument("Null parent thread");
}

ProxyTCPLCListener::ChannelListener::~ChannelListener()
{
   if (psockets != NULL) delete psockets;
   else PSYSTEMLOG(Error, "Null pointer as tcp connected channel");
}

void ProxyTCPLCListener::ChannelListener::Main()
{
   try {
      while ( !parent->isStopped() ) {
         psockets->proxyMsg(ProxyTCPLCListener::LISTENINGTIMEOUT);
      }
   }
   catch (TCPProxyLChannel::ConnectionClose & e) {
      PSYSTEMLOG( Info, "TCP connection for t.120 closed: " << e.what() );
   }
   psockets->s_src.Close();
   psockets->s_dst.Close();
}

