/*
 * Proxy for OpenGate
 * 
 * Copyright (c) Marco Polci
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * $Log: Proxy.h,v $
 * Revision 1.4  2000/12/05 17:17:05  mpolci
 * Added some comments and removed some log message.
 *
 * Revision 1.3  2000/10/19 14:54:24  mpolci
 * Now UDPProxyLChannel::do_proxy throw the exception SocketError on read and write error
 *
 * Revision 1.2  2000/10/17 13:00:13  mpolci
 * File header and CVS log entry setup
 *
 *
 * Revision 1.0  2000/09/04 10:33
 * Initial revision
 *
 */
#ifndef _PROXY_H
#define _PROXY_H

#include <ptlib.h>
#include <ptlib/sockets.h>
#include <stdexcept>
//#include "MyUDPSocket.h"

class ProxyLCListener;

class ProxyLChannel
{
public:
    enum Type { unidirectional, bidirectional };
    enum Protocol { TCP, UDP, RTP, other };
    enum Direction { forward, reverse, both, timeout };
        
    class OpenError : public runtime_error
          {
             public:
                OpenError(const string& what_arg): runtime_error(what_arg) {};
          };

    class BindError : public runtime_error
          {
             public:
                BindError(const string& what_arg): runtime_error(what_arg) {};
          };

    ProxyLChannel(Type d = bidirectional) : fOpened(FALSE) {};
    virtual ~ProxyLChannel() {};

    virtual Protocol getProtocol() = 0;

    virtual void setSource(const PIPSocket::Address & addr, WORD port)
          {    a_src = addr; src_port = port; };
    virtual void setDestination(const PIPSocket::Address & addr, WORD port)
          {    a_dst = addr; dst_port = port; };

    void getSource(PIPSocket::Address & addr, WORD & port) const
          {    addr = a_src; port = src_port; };
    void getDestination(PIPSocket::Address & addr, WORD & port) const
          {    addr = a_dst; port = dst_port; };

    virtual void getSourceSideAddr(PIPSocket::Address & addr, WORD & port) = 0;
    virtual void getDestinationSideAddr(PIPSocket::Address & addr, WORD & port) = 0;

    virtual void bindSockets() throw (BindError) = 0;
    // Task: to binds sockets. It's possible to obtain the source and destination local addresses 
    // only after called this function.

    virtual void open() throw (OpenError) { fOpened = TRUE; };

    virtual ProxyLCListener * allocateListener(PThread::AutoDeleteFlag deletion) = 0;
    // Task: to create a listener for this proxy channel

    //virtual Direction proxyMsg(const PTimeInterval & timeout = 0) = 0;
    // Task: to wait for a messagge and forward it

protected:
    PIPSocket::Address a_src, a_dst;       // endpoint addresses of channel
    WORD               src_port, dst_port; // ...
    BOOL fOpened;
};

////////////////////////////////////////////////////////////////

class UDPProxyLChannel: public ProxyLChannel {
public:
    class SocketError : public runtime_error {
       public:
          SocketError(const string& what_arg): runtime_error(what_arg) {};
    };

    UDPProxyLChannel(Type d = bidirectional);
    virtual ~UDPProxyLChannel();

    virtual Protocol getProtocol() { return UDP; };
    
    virtual void bindSockets();
    // Task: to binds sockets. It's possible to obtain the source and destination local addresses 
    // only after called this function.

    virtual void open();

    virtual ProxyLCListener * allocateListener(PThread::AutoDeleteFlag deletion);

    //virtual Direction proxyMsg(const PTimeInterval & timeout = 0);

    virtual void setSource(const PIPSocket::Address & addr, WORD port);
    virtual void setDestination(const PIPSocket::Address & addr, WORD port);

    virtual void getSourceSideAddr(PIPSocket::Address & addr, WORD & port);
    virtual void getDestinationSideAddr(PIPSocket::Address & addr, WORD & port);

    friend class ProxyUDPLCListener;

protected:
    //MyUDPSocket s_src, s_dst;  // socket to the endpoints
    PUDPSocket s_src, s_dst;  // socket to the endpoints

    // metod for the listener
    Direction proxyMsg(const PTimeInterval & timeout = 0);

private:
    static void do_proxy(PUDPSocket & rd, PUDPSocket & wr);
    void do_bidirectional_proxy();
};

////////////////////////////////////////////////////////////////

class RTCPProxyLChannel;

class RTPProxyLChannel: public UDPProxyLChannel {
public:
   enum Deletion { DeleteRTCP, NoDeleteRTCP };

   RTPProxyLChannel(Type d = bidirectional);
   virtual ~RTPProxyLChannel();

   RTCPProxyLChannel *getRTCP() { return rtcp; };
   void setRTCPDeletion(Deletion d) { fDeleteRTCP = (d == DeleteRTCP); };

   virtual void bindSockets();

protected:
   RTCPProxyLChannel *rtcp;
   BOOL fDeleteRTCP;

private:
   static void bind(PUDPSocket & rtp, PUDPSocket & rtcp);
};

class RTCPProxyLChannel: public UDPProxyLChannel {
public:
   friend class RTPProxyLChannel;

   virtual ~RTCPProxyLChannel();

   RTPProxyLChannel & getRTP() { return *rtp; };
 
   virtual void bindSockets();

protected:
   RTCPProxyLChannel(RTPProxyLChannel * rtp, Type d = bidirectional);

   RTPProxyLChannel * rtp;
};


////////////////////////////////////////////////////////////////

class TCPProxyLChannel: public ProxyLChannel {
public:

    class ConnectionClose : public runtime_error {
       public:
          ConnectionClose(const string& what_arg): runtime_error(what_arg) {};
    };

    struct TCPConnectedChannel {
          PTCPSocket   s_src, s_dst;   // socket to the endpoints
          Direction proxyMsg(const PTimeInterval & timeout = 0);
       private:
          static void do_proxy(PTCPSocket & rd, PTCPSocket & wr);
    };  

    TCPProxyLChannel(Type d = bidirectional);
    virtual ~TCPProxyLChannel();

    virtual Protocol getProtocol() { return TCP; };

    //virtual void open();

    virtual void bindSockets();

    virtual ProxyLCListener * allocateListener(PThread::AutoDeleteFlag deletion);

    //virtual Direction proxyMsg(const PTimeInterval & timeout = 0);

    virtual void getSourceSideAddr(PIPSocket::Address & addr, WORD & port);

    friend class ProxyTCPLCListener;

protected:
    PTCPSocket s_listener;    // to wait for connection from source endpoint

    // metod used by listener
    TCPConnectedChannel * acceptCall();

private:
    // setSource isn't a valid operation for TCPProxyLChannel. Don't use it!
    virtual void setSource(const PIPSocket::Address & addr, WORD port)
        { throw runtime_error("Illegal operation!"); };

    // getDestinationSideAddr isn't a valid operation for TCPProxyLChannel. There are
    // many destination sockets. Don't use it!
    virtual void getDestinationSideAddr(PIPSocket::Address & addr, WORD & port) 
        { addr = 0; port = 0; throw runtime_error("Illegal operation!"); };
};

///////////////////////////////////////////////////////////////

class ProxyLCListener : public PThread
{
    PCLASSINFO(ProxyLCListener, PThread);
public:
    ProxyLCListener(ProxyLChannel &ch, AutoDeleteFlag deletion = AutoDeleteThread);
    ~ProxyLCListener();

    virtual void Main();

    virtual void Listening() = 0;
    // Task: main loop of logical channel listener

    void startListening();
    void terminateListening();

protected:
    ProxyLChannel &plc;
    BOOL fStop, fStart;
};

///////////////////////////////////////////////////////////////

class ProxyUDPLCListener : public ProxyLCListener
{
    PCLASSINFO(ProxyUDPLCListener, ProxyLCListener);
public:
    ProxyUDPLCListener(UDPProxyLChannel & ch, AutoDeleteFlag deletion = AutoDeleteThread);

    virtual void Listening();

protected:
    UDPProxyLChannel & channel() { return static_cast<UDPProxyLChannel &>(plc); };
};

///////////////////////////////////////////////////////////////

class ProxyTCPLCListener : public ProxyLCListener
{
    PCLASSINFO(ProxyTCPLCListener, ProxyLCListener);
public:
    ProxyTCPLCListener(TCPProxyLChannel & ch, AutoDeleteFlag deletion = AutoDeleteThread);

    virtual void Listening();

    BOOL isStopped() { return fStop; };

    static const PTimeInterval LISTENINGTIMEOUT;
    static const PTimeInterval DEATHTIMEOUT;

protected:
    TCPProxyLChannel & channel() { return static_cast<TCPProxyLChannel &>(plc); };

    class ChannelListener: public PThread
    {
          PCLASSINFO(ChannelListener, PThread);
       public:
          ChannelListener( ProxyTCPLCListener * pThrParent,
                           TCPProxyLChannel::TCPConnectedChannel *s, 
                           AutoDeleteFlag deletion = AutoDeleteThread
                         );
          ~ChannelListener();
          virtual void Main();
       protected:
          ProxyTCPLCListener * parent;
          TCPProxyLChannel::TCPConnectedChannel * psockets;
    };
};

#endif
