/* 
 * ProxyTable for OpenGate
 * 
 * Copyright (c) Marco Polci
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * $Log: ProxyTabl.cxx,v $
 * Revision 1.3  2000/12/05 17:17:17  mpolci
 * Added some comments and removed some log message.
 *
 * Revision 1.2  2000/10/17 13:00:13  mpolci
 * File header and CVS log entry setup
 *
 *
 * Revision 1.0  2000/09/04 9:30
 * Initial revision
 *
 */
#include <ptlib.h>
#include <ptlib/svcproc.h>

#include <algorithm>
#include "ProxyTabl.h"

BOOL operator ==( const ProxyTableEntry::ChannelID & id1, const ProxyTableEntry::ChannelID & id2 )
{
   return (id1.lcn == id2.lcn) && (id1.fCallCh == id2.fCallCh); //id1.source == id2.source;
}
         
ProxyTableEntry::ChannelID::ChannelID( const H245_LogicalChannelNumber & num,
                                       bool  fCallerChannel
                                     ) 
   : lcn(num), fCallCh(fCallerChannel)
{
}

ProxyTableEntry::ProxyTableEntry( const H245_LogicalChannelNumber & lcn, 
                                  bool fCallerChannel,
				  ProxyLChannel *ch, 
				  BOOL autodeletechannel
                                )
   : channel(ch), logicalChannelNumber(lcn), 
     fCallCh(fCallerChannel), fDeleteChannel(autodeletechannel)
{
   channel->bindSockets();

   PIPSocket::Address r_a, w_a;
   WORD r_p, w_p;
   try {
     ch->getSourceSideAddr(r_a, r_p);
     ch->getDestinationSideAddr(w_a, w_p);  // not allowed for TCP channels
   }
   catch (...) {};
   PSYSTEMLOG( Info, "New " << (ch->getProtocol() == ProxyLChannel::UDP ? "UDP" : "TCP") << " proxy channel (" 
               << lcn << " " << fCallerChannel << ") src port " << r_p << " dst port " << w_p );

   if (ch == NULL) throw InvalidPLChannel("Null pointer passed as channel");
   listener = ch->allocateListener(PThread::NoAutoDeleteThread);
}

ProxyTableEntry::~ProxyTableEntry()
{
   //PSYSTEMLOG( Info, "ProxyTableEntry::~ProxyTableEntry()");
   stopListener();
   delete listener;
   if (fDeleteChannel) delete channel;
}

ProxyTableEntry::ChannelID ProxyTableEntry::getID() const
{
  return ChannelID( logicalChannelNumber, fCallCh );
}

void ProxyTableEntry::startListener()
// Task: to call after constructed the object and initialized the channel
// to start the proxing of data. Cannot restart after stopped.
{
   //channel->open();  // the listener open the channel before beginning listening
   listener->startListening();
}

void ProxyTableEntry::stopListener()
// Task: to disable the proxing of data through the corresponding channel.
// Cannot restart after stopped
{
   //PSYSTEMLOG( Info, "ProxyTableEntry::stopListener()");
   listener->terminateListening();
   //PSYSTEMLOG( Info, "WaitForTermination ...");
   listener->WaitForTermination();
   //PSYSTEMLOG( Info, "... ok! terminated.");
}

ProxyTable::ProxyTable()
{
}

ProxyTable::~ProxyTable()
{
   while (table.size() > 0) {
      ProxyTableEntry *entry;
      entry = table.front();
      table.pop_front();
      delete entry;
   }
}

ProxyTableEntry & ProxyTable::createNewChannel( const H245_LogicalChannelNumber & logicalChannelNumber, 
                                                bool                              fCallerChannel,
                                                ProxyLChannel::Protocol           proto
                                              )
// Task: to create a new proxy channel associated to the address of the peer who opened logical channel
// and to the logical channel number logicalChannelNumber. The protocol used for the proxy channel is
// indicated by proto. A new entry reppresenting the new channel are created and inserted in the table
// and a reference is returned.
{
   ProxyLChannel *channel = NULL;

   switch (proto) {
      case ProxyLChannel::UDP :  
         channel = new UDPProxyLChannel;
         break;
      case ProxyLChannel::TCP : 
         channel = new TCPProxyLChannel;
         break;
      case ProxyLChannel::RTP : 
         channel = new RTPProxyLChannel;
         break;
      default:
         // the catch section below retrow this exception
         throw NotSupportedProtocolError("Unsupported protocol: cannot create a proxy channel");
         break;
   }
   return createNewChannel(logicalChannelNumber, fCallerChannel, channel);
}

ProxyTableEntry & ProxyTable::createNewChannel( const H245_LogicalChannelNumber & logicalChannelNumber, 
                                                bool                              fCallerChannel,
                                                ProxyLChannel                   * channel
                                              )
// Task: to create a new proxy channel associated to the address of the peer who opened logical channel
// and to the logical channel number logicalChannelNumber. A dinamic allocated object descending of 
// ProxyLChannel are provided as parameter. The proxyTable object frees the memory occupied by channel.
// A new entry reppresenting the new logical channel are created and inserted in the table and a reference 
// is returned.
{
   ProxyTableEntry *entry = NULL;

   try {
      entry = new ProxyTableEntry(logicalChannelNumber, fCallerChannel, channel);
   } catch (...) {
      if (channel != NULL) delete channel;
      if (entry != NULL) delete entry;
      throw;
   }

   tableMutex.Wait();
   try {
      table.push_back(entry);
   } catch (...) {
      tableMutex.Signal();
      throw;
   }
   tableMutex.Signal();

   return *entry;
}

template<class FindStruct> 
void ProxyTable::deleteCh( const FindStruct & f)
{
   tableMutex.Wait();
   table_t::iterator i = find_if( table.begin(), table.end(), f );
   if (i == table.end()) {
      tableMutex.Signal();
      throw NotFoundError("Logical Channel not found");
   } else {
      ProxyTableEntry *entry = *i;
      table.erase(i);
      tableMutex.Signal();
      delete entry;
   }
}

// void ProxyTable::deleteChannel(const ProxyTableEntry::ChannelID & id )
// {
//    tableMutex.Wait();
//    table_t::iterator i = find_if( table.begin(), table.end(),
//                                   findID( id )
//                                 );
//    if (i == table.end()) {
//       tableMutex.Signal();
//       throw NotFoundError("Logical Channel not found");
//    } else {
//       table.erase(i);
//       tableMutex.Signal();
//    }
// }

// void ProxyTable::deleteChannel( ProxyTableEntry * p_entry)
// {
//    tableMutex.Wait();
//    table_t::iterator i = find_if( table.begin(), table.end(),
//                                   findPtr( p_entry )
//                                 );
//    if (i == table.end()) {
//       tableMutex.Signal();
//       throw NotFoundError("Logical Channel not found");
//    } else {
//       table.erase(i);
//       tableMutex.Signal();
//    }
// }

ProxyTableEntry & ProxyTable::find(const ProxyTableEntry::ChannelID & id)
{
   tableMutex.Wait();
   table_t::const_iterator i = find_if( table.begin(), table.end(),
                                        findID(id)
                                      );
   tableMutex.Signal();

   if (i == table.end())
      throw NotFoundError("Logical Channel not found");

   return **i;
}

