/*
 * Proxy Table for OpenGate
 * 
 * Copyright (c) Marco Polci
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * $Log: ProxyTabl.h,v $
 * Revision 1.2  2000/10/17 13:00:13  mpolci
 * File header and CVS log entry setup
 *
 *
 * Revision 1.0  2000/09/05 9:30
 * Initial revision
 *
 */
#ifndef _PROXYTABL_H
#define _PROXYTABL_H

#include <list>
#include <stdexcept>
#include <ptlib.h>
#include <h245.h>
#include "Proxy.h"

using namespace std;

class ProxyTableEntry {
public:
   class InvalidPLChannel : public runtime_error {
      public:
         InvalidPLChannel(const string & what_arg): runtime_error(what_arg) {};
   };

   class ChannelID {
      public:
         //friend class ProxyTableEntry
         friend BOOL operator ==( const ChannelID & , const ChannelID & );

         ChannelID( const H245_LogicalChannelNumber & num,
                    bool fCallerChannel
                  );
      protected:
         H245_LogicalChannelNumber lcn;
         bool fCallCh;
   };

   ProxyTableEntry( const H245_LogicalChannelNumber & lcn,
                    bool fCallerChannel,
                    ProxyLChannel *ch,
                    BOOL autodeletechannel = TRUE
                  );
   ~ProxyTableEntry();

   ChannelID getID() const;
   ProxyLChannel & getChannel() { return *channel; };
   const H245_LogicalChannelNumber & getLogicalChannelNumber() const { return logicalChannelNumber; };

   void startListener();
   void stopListener();

   //friend class ProxyTable;
private:
   ProxyTableEntry() {};                        // Disable default constructor
   ProxyTableEntry(const ProxyTableEntry &) {}; // Disable copy constructor

   ProxyLCListener * listener;
   ProxyLChannel * channel;
   H245_LogicalChannelNumber logicalChannelNumber;
   bool fCallCh;                                  // true if the channel was opened from caller endpoint
   BOOL fDeleteChannel;
};

class ProxyTable {
public:
   class NotFoundError : public runtime_error {
      public:
         NotFoundError(const string & what_arg): runtime_error(what_arg) {};
   };

   class NotSupportedProtocolError : public runtime_error {
      public:
         NotSupportedProtocolError(const string & what_arg): runtime_error(what_arg) {};
   };

   ProxyTable();
   ~ProxyTable();

   ProxyTableEntry & createNewChannel( const H245_LogicalChannelNumber & logicalChannelNumber,
                                       bool                              fCallerChannel,
                                       ProxyLChannel::Protocol           proto
                                     );

   ProxyTableEntry & createNewChannel( const H245_LogicalChannelNumber & logicalChannelNumber,
                                       bool                              fCallerChannel,
                                       ProxyLChannel                   * channel
                                     );
  
   void deleteChannel( const ProxyTableEntry::ChannelID & id ) { deleteCh( findID( id ) ); };
   void deleteChannel( ProxyTableEntry * p)                    { deleteCh( findPtr( p ) ); };
   // Task: to delete the channel indicated by id or pointer to table entry. After this the table entry
   // is deleted so if you use the pointer version of this metod remember don't use the pointer anymore.

   ProxyTableEntry & find( const ProxyTableEntry::ChannelID & id );

   typedef ProxyTableEntry *table_element_t;
private:
   typedef list<table_element_t> table_t;
   table_t table;
   PMutex tableMutex;

   template<class FindStruct> void deleteCh( const FindStruct & );

   struct findID
   {
        ProxyTableEntry::ChannelID id;
        findID(const ProxyTableEntry::ChannelID & cid) : id(cid) {};
	bool operator()( const ProxyTable::table_element_t & entry ) const
	{  return (entry->getID() == id); };
   };

   struct findPtr
   {
        ProxyTable::table_element_t ptr;
        findPtr(ProxyTable::table_element_t p) : ptr(p) {};
	bool operator()( const ProxyTable::table_element_t & entry ) const
	{   return (entry == ptr);        };
   };
};

#endif
