/*
 * Call Table for OpenGate
 *
 * Copyright (c) Egoboo Ltd. 2000
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Open Gatekeeper
 *
 * The Initial Developer of the Original Code is Egoboo Ltd.
 *
 * $Log: CallTabl.h,v $
 * Revision 1.5  2000/05/10 09:18:30  aunitt
 * Removed warning about debug information being truncated in Windows debug build.
 * Removed storage of H.245 target address as we no longer use it.
 *
 * Revision 1.4  2000/05/08 17:48:13  aunitt
 * Added recording of H245 address.
 *
 * Revision 1.3  2000/05/05 11:31:53  aunitt
 * Added bandwidth management.
 *
 * Revision 1.2  2000/04/21 13:46:16  aunitt
 * Removed bogus warnings in Microsoft STL headers.
 *
 * Revision 1.1  2000/04/10 19:21:42  aunitt
 * Call table for OpenGatekeeper.
 *
 *
 */

#ifndef _CALLTABL_H
#define _CALLTABL_H

#include <ptlib.h>
#include <h225.h>
#if (_MSC_VER >= 1200)
#pragma warning( disable : 4786 ) // Disable about debug information being truncated
#pragma warning( push, 3 )
#endif
#include <list>
#include <stdexcept>
#if (_MSC_VER >= 1200)  
#pragma warning( pop )
#endif

using namespace std; 

class Environ;

class CallDetails
{
    public:
        CallDetails( const H225_CallReferenceValue &    CRV,
                     const H225_EndpointIdentifier &    Caller,
                     const H225_EndpointIdentifier &    Called,
                     unsigned                           Bandwidth
                   );
        CallDetails( const H225_CallReferenceValue &    CRV,
                     const H225_CallIdentifier &        Id,
                     const H225_EndpointIdentifier &    Caller,
                     const H225_EndpointIdentifier &    Called,
                     unsigned                           Bandwidth
                   );
        CallDetails() {};
        virtual ~CallDetails();

        const H225_CallReferenceValue & GetCRV() const { return Ref; };
        bool HasIdentifier() const { return HasId; };
        const H225_CallIdentifier & GetId() const { return Id; };
        const H225_EndpointIdentifier & GetCaller() const { return Caller; };
        const H225_EndpointIdentifier & GetCalled() const { return Called; };
        unsigned GetBandwidth() const { return Bandwidth; };
        void SetBandwidth( unsigned NewBandwidth ) { Bandwidth = NewBandwidth; };

    protected:
        //CallDetails() {};	// Stop clients using default constructor

        H225_CallReferenceValue Ref;
        H225_EndpointIdentifier	Caller;
        H225_EndpointIdentifier	Called;
        unsigned                Bandwidth;
        bool                    HasId;      // The call identifier is preferable as it
        H225_CallIdentifier     Id;         // guarenteed to be unique, but it may not exist
};

class CallTable
{
    public:
        CallTable( const Environ & Environ );
        virtual ~CallTable();

        class NotFoundError : public runtime_error
		{
		    public:
			    NotFoundError(const string& what_arg): runtime_error(what_arg) {}
		};
		
		class BandwidthUnavailable : public runtime_error
		{
		    public:
		        BandwidthUnavailable(const string& what_arg): runtime_error(what_arg) {}
		};
		
		class RequestInconsistent : public runtime_error
		{
		    public:
		        RequestInconsistent(const string& what_arg): runtime_error(what_arg) {}
		};
		
    	void Insert( CallDetails & Call );
		// Task: insert the given call into the table
		
		void Remove( const H225_CallIdentifier & Id );
		// Task: to remove the given call from the table by id (prefered version)
		
		void Remove( const H225_CallReferenceValue & CRV );
		// Task: to remove the given call from the table
		
		size_t Size() const { return Table.size(); };
		// Task: to return the number of endpoints in the table

		bool IsEmpty() const { return Table.empty(); };
		// Task: to return true iff the table is empty

		unsigned TotalBandwidthUsed() const { return TotalBandwidth; }
		// Task: returns the total bandwidth used by all calls in the table
		
		H225_BandWidth GetAllowedBandwidth() const;
		// Task: to return the amount of available bandwidth

        CallDetails FindByRef( const H225_CallReferenceValue & CRV );
        // Task: to return the call that has the given reference
        //	 will throw an exception if the call reference doesn't exist

        CallDetails FindByRef( unsigned Reference );
        // Task: to return the call that has the given reference
        //	 will throw an exception if the call reference doesn't exist

        CallDetails FindById( const H225_CallIdentifier & Id );
        // Task; to return the call that has the given call identifier
        //	 will throw an exception if the call identifier doesn't exist

        bool AlterBandwidth( const H225_CallReferenceValue &    CRV,
                             const H225_EndpointIdentifier &    Id,
                                   unsigned                     NewBandwidth
                           );
        // Task: to handle a request to alter the bandwidth of the given call
        //       to the given value. Will return true if the request has been honoured.
        //       Will throw an exception if the call cannot be found or if it doesn't
        //       "belong" to the given endpoint

        bool AlterBandwidth( const H225_CallIdentifier &        CallId,
                             const H225_EndpointIdentifier &    Id,
                                   unsigned                     NewBandwidth
                           );
        // Task: to handle a request to alter the bandwidth of the given call
        //       to the given value. Will return true if the request has been honoured.
        //       Will throw an exception if the call cannot be found or if it doesn't
        //       "belong" to the given endpoint
        //       Similar to the previous call except that it takes a call identifier
        //       This is the preferable version to use if possible
		
    protected:
        CallTable();    // Disallow use of default constructor or copy constructor
        CallTable( const CallTable & );

        typedef list<CallDetails> Table_t;

        void RemoveEntry( Table_t::iterator i );
        // Task: to remove the given entry from the table

        bool AlterBandwidth(       CallDetails                  Call,
                             const H225_EndpointIdentifier &    Id,
                                   unsigned                     NewBandwidth,
                             const string &                     InconsistentError
                           );

        const Environ & MyEnviron;
        Table_t         Table;
   		PMutex          TableMutex;
   		unsigned        TotalBandwidth;
};



#endif // _CALLTABL_H
