/*
 * Call routing thread for OpenGate
 *
 * Copyright (c) Egoboo Ltd. 1999-2000
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * The Original Code is Open Gatekeeper
 *
 * The Initial Developer of the Original Code is Egoboo Ltd.
 *
 * $Log: CallThread.h,v $
 * Revision 1.6  2000/05/23 10:18:49  aunitt
 * Added additional comments.
 *
 * Revision 1.4  2000/05/12 10:21:43  aunitt
 * Added decoding of H.245 messages.
 *
 * Revision 1.3  2000/05/10 09:16:10  aunitt
 * Added H.245 thread to handle routing of H.245 messages.
 *
 * Revision 1.2  2000/05/08 17:45:55  aunitt
 * Added support for recording H245 address.
 *
 * Revision 1.1  2000/04/10 19:22:25  aunitt
 * Call handling thread for gatekeeper call routing for OpenGatekeeper.
 *
 *
 */

#ifndef _CALLTHREAD_H
#define _CALLTHREAD_H

#include <ptlib.h>
#include <ptlib/sockets.h>
#include <ptlib/svcproc.h>
#include <h225.h>
#include <h245.h>
#include "Environ.h"



// This thread handles the dispatching of connections to new signalling threads

class CallThread : public PThread
{
	PCLASSINFO(CallThread, PThread)
	public:
   		CallThread( const Environ & AkaEnviron );
   		
		virtual ~CallThread();

		void Close();	
	
		H225_TransportAddress GetMyCallSignallingAddr();
		// Task: to return the address of our call signalling port
		
		bool IsRunning() const { return Running; };     // Used by children to check when to die
        // Task: to return true iff we are running	
		
	protected:
		CallThread();	                  // Stop clients calling default cons
		CallThread( const CallThread & ); // ...or copy constructor

		virtual void Main();	

		const Environ & MyEnviron;
		PTCPSocket      CallSocket;
		bool            Running;
};

class H245Handler
{
    // This class handles H.245 messages which can either be transmitted on their
    // own TCP connection or can be tunneled in the Q.931 connection
    public:
        H245Handler();

        virtual ~H245Handler();

        virtual void HandleMesg( H245_MultimediaSystemControlMessage & Mesg, bool FromCaller);
        // Task: to handle the given H.245 message

    protected:
        virtual void HandleRequest( H245_RequestMessage & Request, bool FromCaller );
        // Task: to handle the given H.245 request

        virtual void HandleResponse( H245_ResponseMessage & Response, bool FromCaller );
        // Task: to handle the given H.245 response

        virtual void HandleCommand( H245_CommandMessage & Command, bool FromCaller );
        // Task: to handle the given H.245 command

        virtual void HandleIndication( H245_IndicationMessage & Indication, bool FromCaller );
        // Task: to handle the given H.245 indication
};

class H245Thread;

// A signalling thread is created per call channel that goes through us

class SignallingThread : public PThread
{
	PCLASSINFO(SignallingThread, PThread)
	public:
   		SignallingThread( CallThread *      Parent,
   		                  const Environ &   AkaEnviron,
   		                  PTCPSocket *      RecvSocket
   		                );
   		
		virtual ~SignallingThread();
	
	protected:
		SignallingThread();	                            // Stop clients calling default cons
		SignallingThread( const SignallingThread & );   // ...or copy constructor

		virtual void Main();	
		
		void Close();
		
		void ReceiveMesg( bool FromCaller );
        // Task: to receive a Q931 message
	
		void HandleMesg( Q931 & Mesg, bool FromCaller );
        // Task: to handle the given Q931 message 		

        void HandleH245Setup( Q931 & Mesg, H225_H323_UserInformation & UUField );
        // Task: to handle potential setting up of H.245 channel

        // Override the following functions if you want to do fancy
        // things...

		bool HandleSetup( Q931 &                        Mesg,
                                  bool                          FromCaller,
		                  bool                          HasUU,
		                  H225_H323_UserInformation &   UUField
		                );
        // Task: to handle a setup message
        // Retn: true iff message should be forwarded
		
   		bool HandleProceeding( Q931 &                       Mesg,
                                       bool                         FromCaller,
   				               bool                         HasUU,
		                       H225_H323_UserInformation &  UUField
		                     );
        // Task: to handle a call proceeding
        // Retn: true iff message should be forwarded

   		bool HandleAlerting( Q931 &                         Mesg,
                                     bool                           FromCaller,
     				             bool                           HasUU,
		                     H225_H323_UserInformation &    UUField
                           );
        // Task: to handle an alerting message
        // Retn: true iff message should be forwarded

   		bool HandleConnect( Q931 &                      Mesg,
                                    bool                        FromCaller,
   				            bool                        HasUU,
		                    H225_H323_UserInformation & UUField
               	          );
        // Task: to handle a connect message
        // Retn: true iff message should be forwarded

   		bool HandleRelease( Q931 &                      Mesg,
                                    bool                        FromCaller,
   				            bool                        HasUU,
		                    H225_H323_UserInformation & UUField
		                  );
        // Task: to handle a release complete message
        // Retn: true iff message should be forwarded

        bool HandleFacility( Q931 &                         Mesg,
                             bool                           FromCaller,
                             bool                           HasUU,
                             H225_H323_UserInformation &    UUField
                           );
        // Task: to handle a facility message
        // Retn: true iff message should be forwarded

        //void ConnectToDestination( const Q931 & SetupMesg );
        bool ConnectToDestination( const Endpoint & Destination );
        // Task: to connect the socket to the desired destination of the
        //       given setup message

        bool GetCallDetailsForSetup( const Q931 &        SetupMesg, 
                                           CallDetails & MyCall, 
                                           Endpoint &    Destination
                                     );
        // Task: to obtain a call details and destination endpoint relative
        //       to the SetupMesg. Search the CallDetails object in the calls
        //       table and if not found try to create a new call item.

        /*        		
        PIPSocket::Address GetCallerAddress() 
        // Task: return the address of caller peer
        {
                PIPSocket::Address CallerAddr;
                WORD               CallerPort;
                if (CallerSocket != NULL) 
                    CallerSocket->GetPeerAddress( CallerAddr, CallerPort );
                else
                    PSYSTEMLOG(Error, "CallerSocket is NULL");
                return CallerAddr;
        }
        */

		CallThread *    Parent;
		const Environ & MyEnviron;
		PTCPSocket *    CallerSocket;
        PTCPSocket *    CalledSocket;
   		H245Thread *    MyH245Thread;   // To handle routed H.245
   		H245Handler     MyH245Handler;  // To handle tunneled H.245	
        CallDetails     MyCallDetails;
};

// A H245 thread is also created per call that routes H245 through us...

class H245Thread : public PThread, public H245Handler
{
	PCLASSINFO(H245Thread, PThread)
	public:
   		H245Thread( CallThread *                    Parent,
   		            const Environ &                 AkaEnviron,
   		            const H225_TransportAddress &   CalledAddress
                  );
   		
		virtual ~H245Thread();
		
		H225_TransportAddress GetAddress();
		// Task: to return the transport address to give to the caller...
        WORD GetListenerPort();
        // Task: to return the port of the listener socket to give to the caller. 
        //       This metod should replace the previous for multihomed enviroment handling.
	
		void Close();
		// Task: to stop processing of the H245 connection and close it
		
	protected:
		H245Thread();	                    // Stop clients calling default cons
		H245Thread( const H245Thread & );   // ...or copy constructor

		virtual void Main();	
		
		void ReceiveMesg( PTCPSocket * From, PTCPSocket * To );
        // Task: to receive an H.245 message and handle it as appropriate

        void ProcessMesg( H245_MultimediaSystemControlMessage &  Mesg,
			  bool                                   FromCaller,
			  PTCPSocket *                           Target
			);
        // Task: to handle the given H.245 message
/*
        void HandleRequest( H245_RequestMessage & Request );
        // Task: to handle the given H.245 request

        void HandleResponse( H245_ResponseMessage & Response );
        // Task: to handle the given H.245 response
*/
        virtual void HandleCommand( H245_CommandMessage & Command, bool FromCaller );
        // Task: to handle the given H.245 command
/*
        void HandleIndication( H245_IndicationMessage & Indication );
        // Task: to handle the given H.245 indication
*/		
		CallThread *        Parent;
		const Environ &     MyEnviron;
		PTCPSocket          Listener;
		PTCPSocket *        CallerSocket;
        PTCPSocket *        CalledSocket;		
        PIPSocket::Address  CalledAddr;
        bool                SessionEnded;
};


#endif // _CALLTHREAD_H
