/*
 * Proxy Thread for OpenGate
 * 
 * Copyright (c) Marco Polci
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * $Log: ProxyThread.cxx,v $
 * Revision 1.10  2000/12/22 12:12:27  mpolci
 * Modified H245Thread_Proxy::AdmissibleIP. Now the admissible ip can be either
 * the RAS and the call signal address. Before this modification if the RAS address
 * was present the call signal address was not considered valid.
 *
 * Revision 1.9  2000/12/05 17:17:31  mpolci
 * Added some comments and removed some log message.
 *
 * Revision 1.8  2000/12/01 10:26:45  mpolci
 * Added support for multihomed enviroment.
 * Added the metod H245Thread_Proxy::GetLocalAddress().
 * Modified the following metods to use GetLocalAddress() :
 * - H245Thread_Proxy::handleOpenRTP()
 * - H245Thread_Proxy::HandleOpenLogicalChannel()
 * - H245Thread_Proxy::handleRTPAck()
 * - H245Thread_Proxy::handleT120Ack()
 *
 * Revision 1.7  2000/11/24 11:05:23  mpolci
 * Now check Environ.LocalAddr before use it. If equal to INADDR_ANY doesn't use
 * it, retrieves the local address with the PIPSocket::GetHostAddress() function.
 *
 * Revision 1.6  2000/10/25 16:46:12  mpolci
 * Added 2 options in the configuration files to enable ip and port checking of the logical
 * channels for the proxy
 *
 * Revision 1.5  2000/10/25 16:09:24  mpolci
 * Security Fix. Now check the ip address and the port number for the RTP channel.
 *
 * Revision 1.4  2000/10/25 15:25:10  mpolci
 * Security Fix. Now check the port number used for T.120 channel. Accept only non system
 * ports (over 1024). Added metod H245Thread_Proxy::AdmissiblePort()
 *
 * Revision 1.3  2000/10/25 13:33:45  mpolci
 * Security Fix. Added some checks on IP addresses. Now the  store the
 * CallDetails and check the ip addresses of the opened T.120 channel.
 * Modifications:
 * - H245Thread_Proxy (class declaration)
 * - H245Thread_Proxy::MyCall (new member data)
 * - H245Thread_Proxy::AdmissibleIP () (new metod)
 * - H245Thread_Proxy::H245Thread_Proxy()
 * - H245Thread_Proxy::HandleOpenLogicalChannel()
 * - H245Thread_Proxy::handleT120Ack()
 * - H245Thread_Proxy::HandleOpenLogicalChannelConfirm() (removed unused variable warning)
 *
 * Revision 1.2  2000/10/17 13:00:13  mpolci
 * File header and CVS log entry setup
 *
 *
 * Revision 1.0  2000/09/05 14:08
 * Initial revision
 *
 */

#include <ptlib.h>
#include <ptlib/svcproc.h>
#include <h245.h>
#include "ProxyThread.h"


#define CATCH_ALL(msg)  catch (runtime_error & e) { PSYSTEMLOG( Error, msg << ": " << e.what() ); } \
                        catch (logic_error & e) { PSYSTEMLOG( Error, msg << ": " << e.what() ); } \
                        catch ( ... ) { PSYSTEMLOG( Error, msg << ": unknow error" ); } 

void setIPAddress(H245_UnicastAddress_iPAddress & IP, 
                  const PIPSocket::Address & addr, 
                  WORD port) 
{
   for (int i = 0; i < 4; ++i) IP.m_network[i] = addr[i];
   IP.m_tsapIdentifier = port;
}

H245Thread_Proxy::H245Thread_Proxy(CallThread *                    Parent,
                                   const Environ&                  AkaEnviron,
                                   const H225_TransportAddress &   CalledAddress,
                                   const CallDetails &             MyCallDetails
                                  )
    : H245Thread(Parent, AkaEnviron, CalledAddress), MyCall(MyCallDetails)
{
}

H245Thread_Proxy::~H245Thread_Proxy()
{
}

bool H245Thread_Proxy::AdmissibleIP(bool caller, PIPSocket::Address & ip)
// Task: Used on logical channel procedures to check if the ip address used
//       for the logical channel is admissible. The admissible ips are the
//       ip used for RAS addresses or the ip of the call signal addresses
//       of the indicated endpoint (caller or called).
//       If the endpoint is not registered, the given address is ammissible
//       only if the endpoint id is the bogus id and the ip is external to
//       the network covered by the proxy.
{
    if (!MyEnviron.CheckIP())
        return true;

    H225_EndpointIdentifier epId = caller ? MyCall.GetCaller() : MyCall.GetCalled();
    if ( epId == Endpoint::BogusId() ) {
        // Accept only an address external to the network covered by the proxy
        PSYSTEMLOG( Info, ip << MyEnviron.PrxCriteria.isInternal(ip) ? " is internal" : " is external" );
        return !MyEnviron.PrxCriteria.isInternal(ip);
    }

    try {
        Endpoint ep = MyEnviron.EPTable->FindByEndpointId( epId );
        H225_TransportAddress addr;
        H225_TransportAddress_ipAddress IpAddr;
      
        // Check the RAS address
        if ( ep.GetRasIPAddress( addr ) && 
             addr.GetTag() == H225_TransportAddress::e_ipAddress )
        {
            IpAddr = (H225_TransportAddress_ipAddress &) addr;
            PIPSocket::Address ipValid( IpAddr.m_ip[0],
                                        IpAddr.m_ip[1],
                                        IpAddr.m_ip[2],
                                        IpAddr.m_ip[3]
                                      );
            if (ip == ipValid) return true;
        }

        // Check the call signal address
        if ( !ep.GetCallSigIPAddress( IpAddr ) )
                return false;

        PIPSocket::Address ipValid( IpAddr.m_ip[0],
                                    IpAddr.m_ip[1],
                                    IpAddr.m_ip[2],
                                    IpAddr.m_ip[3]
                                  );
        //PSYSTEMLOG(Info, "check ip " << ip);
        //PSYSTEMLOG(Info, "valid ip " << ipValid);
        return ip == ipValid;
    }
    catch (EndpointTable::NotFoundError) 
    {
        PSYSTEMLOG(Error, "Endpoint not found in the table. Cannot accept the ip for proxy operations.");
    }

    return false;
}

bool H245Thread_Proxy::AdmissiblePort( WORD port )
// Task: Used on open logical channel procedures to check if the port number
//       used for the logical channel is admissible. Only non system ports
//       are admissible.
{
    if (!MyEnviron.CheckPortNumber())
        return true;
    else 
        return port > (WORD) 1024;
}

PIPSocket::Address  H245Thread_Proxy::GetLocalAddress(bool CallerSide)
// Task: return the address of the interface to the caller host or the called host.
{
    PIPSocket::Address addr;
    if (CallerSide && CallerSocket != NULL)
        CallerSocket->GetLocalAddress(addr);
    else if (!CallerSide && CalledSocket != NULL)
        CalledSocket->GetLocalAddress(addr);
    else 
        PIPSocket::GetHostAddress(addr);

    //PSYSTEMLOG(Info, "H245Thread_Proxy::GetLocalAddress(" << CallerSide << ") = " << addr);
    return addr;
}

void H245Thread_Proxy::HandleRequest( H245_RequestMessage & Request, bool FromCaller )
// Task: to handle the given H.245 request
{
//    PSYSTEMLOG(Info, "H245Thread_Proxy::HandleRequest()");
    H245Thread::HandleRequest(Request, FromCaller);

    switch( Request.GetTag() )
    {
        case H245_RequestMessage::e_openLogicalChannel  :
            HandleOpenLogicalChannel( Request, FromCaller );
            break;
        case H245_RequestMessage::e_closeLogicalChannel  :
            HandleCloseLogicalChannel( Request, FromCaller );
            break;
        default :
            break;
    }
//    PSYSTEMLOG(Info, "H245Thread_Proxy::HandleRequest() - fine");
}

void H245Thread_Proxy::HandleResponse( H245_ResponseMessage & Response, bool FromCaller )
// Task: to handle the given H.245 response
{
    H245Thread::HandleResponse(Response, FromCaller);

    switch( Response.GetTag() )
    {
        case H245_ResponseMessage::e_openLogicalChannelAck      :
            HandleOpenLogicalChannelAck(Response, FromCaller);
            break;
        case H245_ResponseMessage::e_openLogicalChannelReject   :
            HandleOpenLogicalChannelReject(Response, FromCaller);
            break;
        default :
            break;
    }
}

void H245Thread_Proxy::HandleIndication( H245_IndicationMessage & Indication, bool FromCaller )
// Task: to handle the given H.245 indication
{
    H245Thread::HandleIndication(Indication, FromCaller);

    switch( Indication.GetTag() )
    {
        case H245_IndicationMessage::e_openLogicalChannelConfirm :
            HandleOpenLogicalChannelConfirm(Indication, FromCaller);
            break;
        default :
            break;
    }
}

inline ProxyTableEntry * FIND_ENTRY(ProxyTable & proxy_channels, const ProxyTableEntry::ChannelID & id)
{
   try {
      return & proxy_channels.find( id );
   }
   catch ( ProxyTable::NotFoundError & e ) {
      return NULL;
   }
}


BOOL H245Thread_Proxy::handleOpenRTP( H245_TransportAddress &           Addr,
                                      const H245_LogicalChannelNumber & flcNum,
                                      bool                              FromCaller,
                                      bool                              fControlChannel
                                    )
{
   if ( Addr.GetTag() == H245_TransportAddress::e_unicastAddress )
   {
      H245_UnicastAddress & UnicastAddr = Addr;
      if ( UnicastAddr.GetTag() == H245_UnicastAddress::e_iPAddress )
      {
         H245_UnicastAddress_iPAddress & IP = UnicastAddr;
         PIPSocket::Address IPAddr(IP.m_network[0],
                                   IP.m_network[1],
                                   IP.m_network[2],
                                   IP.m_network[3]
                                  );
         
         ProxyTableEntry * p_entry_media,
                         * p_entry_control;

         p_entry_media = FIND_ENTRY( proxy_media_channels, 
                                     ProxyTableEntry::ChannelID(flcNum, FromCaller) 
                                   );
         p_entry_control = FIND_ENTRY( proxy_control_channels, 
                                       ProxyTableEntry::ChannelID(flcNum, FromCaller) 
                                     );
         if (p_entry_media == NULL) {
            // Must create a new RTP proxy channel
            if ( p_entry_control != NULL ) {
               PSYSTEMLOG( Error, 
                         "Cannot create a new media channel. The corresponding control channel already exists."
                         );
               return FALSE;
            }
            try {
               // create rtp channel ( media channel )
               p_entry_media = & proxy_media_channels.createNewChannel( flcNum, FromCaller, 
                                                                        ProxyLChannel::RTP 
                                                                      );
               // create corresponding rtcp channel ( control channel )
               RTPProxyLChannel & rtp = dynamic_cast<RTPProxyLChannel &>( p_entry_media->getChannel() );
               p_entry_control = & proxy_control_channels.createNewChannel( flcNum, FromCaller,
                                                                            rtp.getRTCP()
                                                                          );
               rtp.setRTCPDeletion( RTPProxyLChannel::NoDeleteRTCP );
            }
            catch ( ... ) {
               PSYSTEMLOG( Error, "Cannot create RTP and RTCP channels" );
               if (p_entry_media != NULL) {
                  try { proxy_media_channels.deleteChannel( p_entry_media ); }
                  catch (...) {
                     PSYSTEMLOG(Error, "Abnormal situation: RTP created but cannot delete it from table");
                  }
                  //PSYSTEMLOG( Info, "media channel (" << flcNum << " " << FromCaller << ") deleted");
               } 
               return FALSE;
            }
         } else {
            // the media channel already exists and the control channel must exists too.
            if ( p_entry_control == NULL ) {
               PSYSTEMLOG( Error, "The media channel exists but the corresponding control channel doesn't.");
               return FALSE;
            }
         }

         // Ok! either media and control channel exist

         // Security check. Check if the given ip belongs to the endpoint and if the
         // port number is not a system port.
         if ( AdmissibleIP(FromCaller, IPAddr) && AdmissiblePort(IP.m_tsapIdentifier) )
         {
             // Store channel opener addresses and modify H.245 message fields with proxy addresses.
             PIPSocket::Address proxyIP;
             WORD proxyPort;

             if (fControlChannel) {
                 p_entry_control->getChannel().setSource(IPAddr, IP.m_tsapIdentifier);
                 p_entry_control->getChannel().getDestinationSideAddr(proxyIP, proxyPort);
             } else {
                 p_entry_media->getChannel().setSource(IPAddr, IP.m_tsapIdentifier);
                 p_entry_media->getChannel().getDestinationSideAddr(proxyIP, proxyPort);
             }
//              if (MyEnviron.LocalAddr == INADDR_ANY)
//                  PIPSocket::GetHostAddress(proxyIP);
//              else
//                  proxyIP = MyEnviron.LocalAddr;
//              setIPAddress(IP, proxyIP, proxyPort);
              setIPAddress(IP, GetLocalAddress(!FromCaller), proxyPort);

             p_entry_media->startListener();
             p_entry_control->startListener();
         } else {
             PSYSTEMLOG( Error, "Security alert - proxy channel (" << flcNum 
                                 << " - " << FromCaller << ") disabled"
                       );
         }

         return TRUE;
      }
   }
   return FALSE;
}



// BOOL H245Thread_Proxy::handleOpen( H245_TransportAddress &           Addr, 
//                                    const H245_LogicalChannelNumber & FLCNum, 
//                                    bool                              FromCaller,
//                                    ProxyTable &                      proxy_channels
//                                  )
// {
//    PSYSTEMLOG( Error, "Questa funzione non deve essere chiamata");
//    if ( Addr.GetTag() == H245_TransportAddress::e_unicastAddress )
//    {
//       H245_UnicastAddress & UnicastAddr = Addr;
//       if ( UnicastAddr.GetTag() == H245_UnicastAddress::e_iPAddress )
//       {
//          H245_UnicastAddress_iPAddress & IP = UnicastAddr;
//          PIPSocket::Address IPAddr(IP.m_network[0],
//                                    IP.m_network[1],
//                                    IP.m_network[2],
//                                    IP.m_network[3]
//                                   );
//          //PSYSTEMLOG( Info, "HandleOpenLogicalChannel media channel (" 
//          //            <<  olc.m_forwardLogicalChannelNumber << " - " << FromCaller << ")"
//          //          );
//          try {
//             ProxyTableEntry & entry = proxy_channels.createNewChannel( FLCNum,
//                                                                        FromCaller, ProxyLChannel::UDP
//                                                                      );
//             entry.getChannel().setSource(IPAddr, IP.m_tsapIdentifier);

//             entry.startListener();

//             PIPSocket::Address proxyIP;
//             WORD proxyPort;
//             entry.getChannel().getDestinationSideAddr(proxyIP, proxyPort);
//             proxyIP = MyEnviron.LocalAddr;

//             setIPAddress(IP, proxyIP, proxyPort);

//             // PSYSTEMLOG(Info, "New IP :\n" << IP);

//          } catch (runtime_error &e) {
//             PSYSTEMLOG(Error, e.what());
//          }
//          return TRUE;
//       }
//    }
//    return FALSE;
// }

BOOL isT120SeparateLANStack( H245_DataType & data_type )
{
   if (data_type.GetTag() == H245_DataType::e_data )
   { 
      H245_DataApplicationCapability & cap = data_type;

      if ( cap.m_application.GetTag() == H245_DataApplicationCapability_application::e_t120 )
      {
         H245_DataProtocolCapability & proto_cap = cap.m_application;

         if ( proto_cap.GetTag() == H245_DataProtocolCapability::e_separateLANStack )
         {
            // OK! This is a T.120 connection over a TCP connection.
            return TRUE;
         }
      }
   }
   return FALSE;
}

void H245Thread_Proxy::HandleOpenLogicalChannel( H245_OpenLogicalChannel & olc, bool FromCaller )
{
   H245_LogicalChannelNumber & flcnum = olc.m_forwardLogicalChannelNumber;

   // For T120
   if ( isT120SeparateLANStack( olc.m_forwardLogicalChannelParameters.m_dataType) )
   {
      // The forward channel is a T.120 connection over a TCP connection.
      // ... and the reverse channel ?
      if ( olc.HasOptionalField(H245_OpenLogicalChannel::e_reverseLogicalChannelParameters) )
      {
         if ( isT120SeparateLANStack(olc.m_reverseLogicalChannelParameters.m_dataType) )
         {
            // ... also the reverse channel is a T.120 connection over a TCP connection.
            
            // Create a new TCP proxy channel
            try {
               ProxyTableEntry & entry = proxy_t120_channels.createNewChannel( flcnum,
                                                                               FromCaller, 
                                                                               ProxyLChannel::TCP
                                                                             );
               PSYSTEMLOG( Info, "HandleOpenLogicalChannel: T120 channel (" 
                                 <<  flcnum << " - " << FromCaller << ")"
                         );

               if ( olc.HasOptionalField(H245_OpenLogicalChannel::e_separateStack) )
               {
                  H245_NetworkAccessParameters & net_acc_param = olc.m_separateStack;
                  
                  if ( net_acc_param.m_networkAddress.GetTag() == 
                       H245_NetworkAccessParameters_networkAddress::e_localAreaAddress
                     )
                  {
                     H245_TransportAddress & Addr = net_acc_param.m_networkAddress;
                     // ...
                     if ( Addr.GetTag() == H245_TransportAddress::e_unicastAddress )
                     {
                        H245_UnicastAddress & UnicastAddr = Addr;
                        if ( UnicastAddr.GetTag() == H245_UnicastAddress::e_iPAddress )
                        {
                           H245_UnicastAddress_iPAddress & IP = UnicastAddr;
                           PIPSocket::Address IPAddr( IP.m_network[0],
                                                      IP.m_network[1],
                                                      IP.m_network[2],
                                                      IP.m_network[3]
                                                    );
                           //PSYSTEMLOG( Info, "HandleOpenLogicalChannel T.120 channel (" 
                           //            <<  olc.m_forwardLogicalChannelNumber << " - " << FromCaller << ")"
                           //          );

                           // Security check. Check if the given ip belongs to the endpoint and if the
                           // port number is not a system port.
                           if ( AdmissibleIP(FromCaller, IPAddr) && AdmissiblePort(IP.m_tsapIdentifier) ) {
                               entry.getChannel().setDestination(IPAddr, IP.m_tsapIdentifier);

                               PIPSocket::Address proxyIP;
                               WORD proxyPort;
                               entry.getChannel().getSourceSideAddr(proxyIP, proxyPort);
//                                if (MyEnviron.LocalAddr == INADDR_ANY)
//                                    PIPSocket::GetHostAddress(proxyIP);
//                                else
//                                    proxyIP = MyEnviron.LocalAddr;
//                                setIPAddress(IP, proxyIP, proxyPort);
                               setIPAddress(IP, GetLocalAddress(!FromCaller), proxyPort);
                           } else {
                               PSYSTEMLOG( Error, "Security alert - proxy channel (" 
                                           <<  olc.m_forwardLogicalChannelNumber << " - " << FromCaller 
                                           << ") disabled"
                                         );
                           }
                        }
                     }
                  } else {
                     // ??? error ??? Not a localAreaAddress
                  }
               }
            } catch (runtime_error &e) {
               PSYSTEMLOG(Error, e.what());
            }
         }
      }
   }

   // get multiplexParameters field
   H245_OpenLogicalChannel_forwardLogicalChannelParameters_multiplexParameters & MultiParams =
        olc.m_forwardLogicalChannelParameters.m_multiplexParameters;

   if ( MultiParams.GetTag() == 
        H245_OpenLogicalChannel_forwardLogicalChannelParameters_multiplexParameters::e_h2250LogicalChannelParameters
      )
   {
      // multiplexParameters is a h2250LogicalChannelParameters
      H245_H2250LogicalChannelParameters & H225Params = MultiParams;

      // handle the control channel if present
      if ( H225Params.HasOptionalField( H245_H2250LogicalChannelParameters::e_mediaControlChannel ) )
      {
         H245_TransportAddress & Addr = H225Params.m_mediaControlChannel;

         if ( handleOpenRTP(Addr, olc.m_forwardLogicalChannelNumber, FromCaller, TRUE))
         {          
            PSYSTEMLOG( Info, "HandleOpenLogicalChannel: media control channel (" 
                           <<  olc.m_forwardLogicalChannelNumber << " - " << FromCaller << ")"
                      );
            //return;
         }
      }


      // handle the media channel if present 
      if ( H225Params.HasOptionalField( H245_H2250LogicalChannelParameters::e_mediaChannel ) )
      {
         H245_TransportAddress & Addr = H225Params.m_mediaChannel;

         if ( handleOpenRTP(Addr, olc.m_forwardLogicalChannelNumber, FromCaller, FALSE) )
         {          
            PSYSTEMLOG( Info, "HandleOpenLogicalChannel: media channel (" 
                        <<  olc.m_forwardLogicalChannelNumber << " - " << FromCaller << ")"
                      );
            //return;
         }
      }
   }
   //PSYSTEMLOG( Info, "H323 Proxy: cannot handle channel " << olc.m_forwardLogicalChannelNumber 
   //                  << " - " << FromCaller
   //          );
}

BOOL H245Thread_Proxy::handleRTPAck( H245_TransportAddress &           Addr, 
                                  const H245_LogicalChannelNumber & flcNum, 
                                  bool                              FromCaller,
                                  ProxyTable &                      proxy_channels
                                )
{
   if ( Addr.GetTag() == H245_TransportAddress::e_unicastAddress )
   {
      H245_UnicastAddress & UnicastAddr = Addr;
      if ( UnicastAddr.GetTag() == H245_UnicastAddress::e_iPAddress )
      {
         H245_UnicastAddress_iPAddress & IP = UnicastAddr;
         PIPSocket::Address IPAddr(IP.m_network[0],
                                   IP.m_network[1],
                                   IP.m_network[2],
                                   IP.m_network[3]
                                  );
         try {
            // Search logical channel in the table
            ProxyTableEntry & entry = 
                          proxy_channels.find( ProxyTableEntry::ChannelID(flcNum, !FromCaller) );
            PSYSTEMLOG( Info, "Ack - Channel found " << flcNum << " - " << !FromCaller 
                        << " destination " << IPAddr << " port " << IP.m_tsapIdentifier);
            
            // Security check. Check if the given ip belongs to the endpoint and if the
            // port number is not a system port.
            if ( AdmissibleIP(FromCaller, IPAddr) && AdmissiblePort(IP.m_tsapIdentifier) ) {
                // Store endpoint address and modify H.245 message fields with proxy addresses.
                entry.getChannel().setDestination(IPAddr, IP.m_tsapIdentifier);
                PIPSocket::Address proxyIP;
                WORD proxyPort;
                entry.getChannel().getSourceSideAddr(proxyIP, proxyPort);
//                 if (MyEnviron.LocalAddr == INADDR_ANY)
//                     PIPSocket::GetHostAddress(proxyIP);
//                 else
//                     proxyIP = MyEnviron.LocalAddr;
//                 setIPAddress(IP, proxyIP, proxyPort);
              setIPAddress(IP, GetLocalAddress(!FromCaller), proxyPort);
            } else {
                PSYSTEMLOG( Error, "Security alert - proxy channel (" << flcNum
                                   << " - " << FromCaller << ") disabled"
                          );
            }

            return TRUE;

         } 
         catch ( ProxyTable::NotFoundError & e ) {
            // The channel does not exists.
            PSYSTEMLOG(Error, "Ack - Error! Channel dosn't exist (" << flcNum << " - " << !FromCaller 
                       << "). Destination " << IPAddr << " port " << IP.m_tsapIdentifier);
         }
         catch (runtime_error & e) {
            PSYSTEMLOG(Error, e.what());
         }
      }
   }
   return FALSE;
}

BOOL H245Thread_Proxy::handleT120Ack( H245_TransportAddress &           Addr, 
                                      const H245_LogicalChannelNumber & flcNum, 
                                      bool                              FromCaller
                                    )
{
   if ( Addr.GetTag() == H245_TransportAddress::e_unicastAddress )
   {
      H245_UnicastAddress & UnicastAddr = Addr;
      if ( UnicastAddr.GetTag() == H245_UnicastAddress::e_iPAddress )
      {
         H245_UnicastAddress_iPAddress & IP = UnicastAddr;
         PIPSocket::Address IPAddr(IP.m_network[0],
                                   IP.m_network[1],
                                   IP.m_network[2],
                                   IP.m_network[3]
                                  );
         try {
            ProxyTableEntry & entry = 
                      proxy_t120_channels.find( ProxyTableEntry::ChannelID(flcNum, !FromCaller) );
            // Security check. Check if the given ip belongs to the endpoint and if the
            // port number is not a system port.
            if ( AdmissibleIP(FromCaller, IPAddr) && AdmissiblePort(IP.m_tsapIdentifier) ) {
               entry.getChannel().setDestination(IPAddr, IP.m_tsapIdentifier);

               PSYSTEMLOG( Info, "T120 channel found " << flcNum << " - " << !FromCaller
                           << " destination " << IPAddr << " port " << IP.m_tsapIdentifier);

               PIPSocket::Address proxyIP;
               WORD proxyPort;
               entry.getChannel().getSourceSideAddr(proxyIP, proxyPort);
//                if (MyEnviron.LocalAddr == INADDR_ANY)
//                    PIPSocket::GetHostAddress(proxyIP);
//                else
//                    proxyIP = MyEnviron.LocalAddr;
//                setIPAddress(IP, proxyIP, proxyPort);
              setIPAddress(IP, GetLocalAddress(!FromCaller), proxyPort);

               entry.startListener();
            } else {
               PSYSTEMLOG( Error, "Security alert - proxy channel disabled");
            }
            return TRUE;

         } catch ( ProxyTable::NotFoundError & e ) {
            PSYSTEMLOG(Info, "T120 channel not found " << flcNum << " - " << !FromCaller);
         } catch (runtime_error &e) {
            PSYSTEMLOG(Error, e.what());
         }
      }
   }
   return FALSE;
}

void H245Thread_Proxy::HandleOpenLogicalChannelAck(H245_OpenLogicalChannelAck & msg, bool FromCaller )
{
   H245_LogicalChannelNumber & flcNum = msg.m_forwardLogicalChannelNumber;

   // For T120
   if ( msg.HasOptionalField(H245_OpenLogicalChannelAck::e_separateStack) )
   {
      H245_NetworkAccessParameters & net_acc_param = msg.m_separateStack;
                  
      if ( net_acc_param.m_networkAddress.GetTag() == 
           H245_NetworkAccessParameters_networkAddress::e_localAreaAddress
         )
      {
         H245_TransportAddress & Addr = net_acc_param.m_networkAddress;

         if ( handleT120Ack(Addr, flcNum, FromCaller) )
         {          
            PSYSTEMLOG( Info, "HandleOpenLogicalChannelAck: T.120 channel (" 
                        << flcNum << " - " << !FromCaller << ")"
                      );
         }
      } else {
         // ??? error ???
      }
   } else {
      try {
         ProxyTableEntry & entry = 
             proxy_t120_channels.find( ProxyTableEntry::ChannelID(flcNum, !FromCaller) );

         PSYSTEMLOG( Info, "Ack - T120 channel found " << flcNum << " - " << !FromCaller 
                           << " No separateStack provided"
                   );
         entry.startListener();

         return;
      } catch ( ProxyTable::NotFoundError & e ) {
        // OK. This isn't a T.120 channel.
      } catch (runtime_error &e) {
        PSYSTEMLOG(Error, e.what());
      }
   }

   // For media channels
   if ( msg.HasOptionalField(H245_OpenLogicalChannelAck::e_forwardMultiplexAckParameters) )
   {
      H245_OpenLogicalChannelAck_forwardMultiplexAckParameters & ackparam = 
                                      msg.m_forwardMultiplexAckParameters;
      if (ackparam.GetTag() == 
          H245_OpenLogicalChannelAck_forwardMultiplexAckParameters::e_h2250LogicalChannelAckParameters
         )
      {
         H245_H2250LogicalChannelAckParameters & h225Params = ackparam;
         // Control channel

         if ( h225Params.HasOptionalField( H245_H2250LogicalChannelAckParameters::e_mediaControlChannel ) )
         {
            H245_TransportAddress & Addr = h225Params.m_mediaControlChannel;

            if ( handleRTPAck(Addr, flcNum, FromCaller, proxy_control_channels) )
            {
               /*
               PSYSTEMLOG( Info, "HandleOpenLogicalChannelAck: media control channel (" 
                                 << flcNum << " - " << !FromCaller << ")"
                         );
               */
            }
         }


         // Media channel
         if ( h225Params.HasOptionalField( H245_H2250LogicalChannelAckParameters::e_mediaChannel ) )
         {
            H245_TransportAddress & Addr = h225Params.m_mediaChannel;

            if ( handleRTPAck(Addr, flcNum, FromCaller, proxy_media_channels) )
            {    
               /*      
               PSYSTEMLOG( Info, "HandleOpenLogicalChannelAck: media channel (" 
                                 << flcNum << " - " << !FromCaller << ")"
                         );
               */
            }
         }
      }
   }
}

void H245Thread_Proxy::HandleOpenLogicalChannelConfirm( H245_OpenLogicalChannelConfirm & msg, bool FromCaller )
{
   try {
      H245_LogicalChannelNumber & flcnum = msg.m_forwardLogicalChannelNumber;

      try {
         // ProxyTableEntry & entry = 
              proxy_t120_channels.find( ProxyTableEntry::ChannelID(flcnum, FromCaller) );

         PSYSTEMLOG( Info, "Confirm - T120 channel found " << flcnum << " - " << FromCaller );
         return;

      } catch ( ProxyTable::NotFoundError & e ) {
         // OK. This isn't a T.120 channel.
      }
   
      PSYSTEMLOG( Info, "openLogicalChannelConfirm " << flcnum << " - " << FromCaller );
   }
   CATCH_ALL("HandleOpenLogicalChannelConfirm()");
}


void H245Thread_Proxy::HandleOpenLogicalChannelReject( H245_OpenLogicalChannelReject & msg, bool FromCaller )
{
   PSYSTEMLOG( Info, "openLogincalChannelReject (" << msg.m_forwardLogicalChannelNumber 
               << "-" << !FromCaller << ")" );
   try {
      deleteChannel( msg.m_forwardLogicalChannelNumber, !FromCaller );
   }
   CATCH_ALL("HandleOpenLogicalChannelReject()");
}

void H245Thread_Proxy::HandleCloseLogicalChannel( H245_CloseLogicalChannel & msg, bool FromCaller )
{
   PSYSTEMLOG( Info, "closeLogicalChannel (" << msg.m_forwardLogicalChannelNumber 
               << "-" << FromCaller << ")" );
   try {
      deleteChannel( msg.m_forwardLogicalChannelNumber, FromCaller );
   }
   CATCH_ALL("HandleCloseLogicalChannel()");
}

void H245Thread_Proxy::deleteChannel( const H245_LogicalChannelNumber & flcnum, bool FromCaller )
{
   // is a T.120 channel? if yes delete it.
   try {
      proxy_t120_channels.deleteChannel( ProxyTableEntry::ChannelID(flcnum, FromCaller) );
      
      PSYSTEMLOG( Info, "T.120 channel deleted (" << flcnum << " - " << FromCaller << ")");
      return;
   } 
   catch ( ProxyTable::NotFoundError & e ) {}  // OK. This isn't a T.120 channel.
   
   // isn't a t.120 channel, brobably it's a RTP channel. If so delete it from media and control tables.
   try {
      proxy_media_channels.deleteChannel( ProxyTableEntry::ChannelID(flcnum, FromCaller) );
      
      PSYSTEMLOG( Info, "RTP channel deleted (" << flcnum << " - " << FromCaller << ")");
   }
   catch ( ProxyTable::NotFoundError & e ) {  // This isn't a RTP channel ???
      PSYSTEMLOG( Warning, "the channel (" << flcnum << " - " << FromCaller
                           << ") isn't a t.120 neither a rtp channel");
   }
   CATCH_ALL("HandleOpenLogicalChannelReject()");

   try {
      proxy_control_channels.deleteChannel( ProxyTableEntry::ChannelID(flcnum, FromCaller) );

      PSYSTEMLOG( Info, "RTCP channel deleted (" << flcnum << " - " << FromCaller << ")");
      return;
   } 
   catch ( ProxyTable::NotFoundError & e ) {  // This isn't a RTCP channel ???
      PSYSTEMLOG( Warning, "The channel (" << flcnum << " - " << FromCaller
                            << ") is not a rtcp channel");
   }
}

