/*
 * RouteTable for OpenH323Proxy
 * 
 * Copyright (c) Marco Polci
 *
 * The contents of this file are subject to the Mozilla Public License
 * Version 1.0 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific language governing rights and limitations
 * under the License.
 *
 * $Log: RouteTbl.cxx,v $
 * Revision 1.2  2000/12/01 11:32:16  mpolci
 * Added the const qualifier to the metod RouteTbl::GetLocalAddress()
 *
 * Revision 1.1  2000/11/28 12:01:38  mpolci
 * Initial revision.
 * Definition of the class RouteTbl. The class stores the route table of
 * the host running the proxy. The main utilization is to obtain the local
 * address visible from a specified host. This is to resolve the multi-homed
 * enviroment problems.
 *
 *
 */

#include <ptlib.h>
#include <ptlib/svcproc.h>
#include "RouteTbl.h"

RouteTbl::RouteTbl()
{
    initTable();
}

RouteTbl::~RouteTbl()
{
}


PIPSocket::Address RouteTbl::GetLocalAddress(PIPSocket::Address addr) const
// Task: to get the local address associated to the interface used to route the
//       specified address.
{
    PINDEX routes = table.GetSize();
    PINDEX r=0;

    while (r < routes) {
        TblEntry entry = table[r];
        //PSYSTEMLOG(Info, "route entry net=" << entry.network << " mask=" 
        //           << entry.net_mask << " addr=" << entry.local_addr);
        if ( (addr & entry.net_mask) == entry.network ) 
            return entry.local_addr;
        else 
            ++r;
    }
    return GetLocalAddress();
}

static bool getInterfaceAddress(const PIPSocket::InterfaceTable & if_table, 
                                const PString                     ifname,
                                      PIPSocket::Address &        addr
                               )
{
    PINDEX interfaces = if_table.GetSize();
    PINDEX i=0;

    while (i < interfaces) {
        PIPSocket::InterfaceEntry if_entry = if_table[i];
        if ( if_entry.GetName() == ifname ) {
            addr = if_entry.GetAddress();
            return true;
        } else
            ++i;
    }
    return false;
}


class initError {};
//typedef int initError;

void RouteTbl::initTable()
{
    bool fDefGW = false;
    try {
        PIPSocket::RouteTable r_table;
        PIPSocket::InterfaceTable if_table;
        PINDEX routes;

        if ( !PIPSocket::GetInterfaceTable( if_table ) ) {
            PSYSTEMLOG(Info, "no interface table" );
            throw initError();
        }
        // get default gw info
        PString gw_if = PIPSocket::GetGatewayInterface();
        if ( gw_if.IsEmpty() )
            PSYSTEMLOG(Warning, "No default gateway");
        else 
        if ( !getInterfaceAddress(if_table, gw_if, localAddr) ) {
            PSYSTEMLOG( Error, "Interface for default gateway not found: " << gw_if );
            if ( PIPSocket::GetHostAddress(localAddr) )
                fDefGW = true;
            else 
                PSYSTEMLOG( Error, "Cannot obtain local host address" );
        }
        else
            fDefGW = true;
        // get route table
        if ( !PIPSocket::GetRouteTable( r_table ) ) {
       	    PSYSTEMLOG(Error, "no route table" );
            throw initError();
        }

        routes = r_table.GetSize();
        // build my route table
	for (PINDEX r=0; r < routes; r++) {
            PIPSocket::RouteEntry r_entry = r_table[r];

            PIPSocket::Address laddr;
            if ( getInterfaceAddress(if_table, r_entry.GetInterface(), laddr) )
                table.Append( new TblEntry(r_entry.GetNetwork(), 
                                      r_entry.GetNetMask(), laddr)
                           );
            else
                PSYSTEMLOG( Error, "Interface not found: " << r_entry.GetInterface() );
        }
    } 
    catch (initError) {}

    // add default route at the end of the list
    if (fDefGW)
        table.Append( new TblEntry(0, 0, localAddr) );

}
