package PGDb;
use strict;
use vars qw($AUTOLOAD);
use Carp qw(confess);

#    pgrsync - Program to syncronise/copy tables between PostgreSQL databases
#    Copyright (C) 2002  Martijn van Oosterhout
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use Pg;

sub new ($)
{
  my $class = shift;
  my $str = shift;

  my $db = Pg::connectdb( $str );

  if( $db->status != PGRES_CONNECTION_OK )
  {
    die "Connection failure: ".$db->errorMessage;
  }
  my $res = $db->exec( "select version()" );
  my $ver = $res->getvalue(0,0);
  my $version;
  if( $ver =~ /postgresql (\d)\.(\d)\.(\d)/i )
  {
    $version = "$1.$2";
  }
  else
  {
    die "Couldn't parse version ($ver)\n";
  }
  return bless { db => $db, version => $version }, $class;
}

sub version ()
{
  return shift->{version};
}

sub getTables ()
{
  my $self = shift;

  my $res = $self->{db}->exec( "select relname from pg_class where relkind = 'r'" );

  return map { $res->getvalue($_,0) } (0..$res->ntuples-1);
}

sub getTable ($)
{
  my $self = shift;

  return PGDb::Table->new( $self, @_ );
}

sub db
{
  return shift->{db};
}

sub errorMessage ()
{
  shift->{db}->errorMessage;
}

package PGDb::Table;

sub new ($)
{
  my $class = shift;
  my $pgdb  = shift;
  my $table = shift;

  my $res = $pgdb->db->exec( "select oid from pg_class where relname = '$table' and relkind = 'r'" );

  if( $res->ntuples == 0 )
  { return undef }

  my $oid = $res->getvalue(0,0);

  return bless { db => $pgdb, oid => $oid }, $class;
}

sub db
{
  return shift->{db}->db;
}

sub getAttributes ()
{
  my $self = shift;

  my $res = $self->db->exec( "select attname from pg_attribute where attrelid = $self->{oid} and attnum > 0 order by attnum" );

  return map { $res->getvalue($_,0) } (0..$res->ntuples-1);
}

sub getAttribute ($)
{
  my $self = shift;

  return PGDb::Attribute->new( $self, @_ );
}

# We're looking for a unique index on a single column. Return to column index (0=first)
sub getPkeyIndex ()
{
  my $self = shift;

  my $res = $self->db->exec( "select indkey from pg_index where indrelid = $self->{oid} and indisunique = 't';" );

  my $pkindex = 999;

  # No unique indexes, what can we do?
  if( $res->ntuples == 0 )
  {
    return undef;
  }

  for my $i (0..$res->ntuples-1)
  {
    my @list = split / /, $res->getvalue($i,0);
    next if( @list > 1 );                        # Only single key supported for now
    next if( scalar( grep { $_ <= 0 } @list ) ); # Only real columns thanks

    if( $list[0] < $pkindex ) { $pkindex = $list[0] }
  }
  # No suitable indexes found
  if( $pkindex == 999 )
  {
    return undef;
  }
  return $pkindex - 1;            # zero offset
}

package PGDb::Attribute;

sub new
{
  my $class = shift;
  my $pgtable = shift;
  my $attr = shift;

  my $res = $pgtable->db->exec( "select atttypid, attnotnull, atthasdef from pg_attribute where attrelid = $pgtable->{oid} and attname = '$attr'" );

  if( $res->ntuples == 0 )
  { return undef }

  my ($atttypid, $attnotnull, $atthasdef) = $res->getvalue(0,0);

  return bless { table => $pgtable, typeoid => $atttypid, notnull => $attnotnull, hasdefault => $atthasdef }, $class;
}

sub db
{
  return shift->{table}->db;
}


1;
