#!/usr/bin/perl -w

#    pgrsync - Program to syncronise/copy tables between PostgreSQL databases
#    Copyright (C) 2002  Martijn van Oosterhout
#
#    This program is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    This program is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with this program; if not, write to the Free Software
#    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use strict;
use lib 'perl/pgrsync';

use Getopt::Long; # qw(:config gnu_getopt auto_abbrev);
use Data::Dumper;

sub Encode ($$);
sub Decode ($$);
sub MakeConnection($);

my $PROTO = '1.0';
my %msgtypes = (
  'V'  =>  's',                  # Init (or identify) (to get version)
  'E'  =>  's',                  # Error message
  'B'  =>  's',                  # Begin checksums for table
  'C'  =>  'si',                 # Plain checksum
  'F'  =>  '',                   # Finish table
  'A'  =>  '',                   # Abort table transfer (either src or dst)
  'T'  =>  'si(s)',              # Table descriptor
  'S'  =>  '(s)',                # Setup table list
  's'  =>  'ci',                 # Setup (type,seed)
  'U'  =>  's(s)',               # Update record
  'I'  =>  's(s)',               # Insert record
  'D'  =>  's',                  # Delete record
  'Q'  =>  '',                   # Done
);

#my $a;
#
#foreach my $i (0..15)
#{
#  $a .= '\0';
#  print "$i => ",(map { sprintf("%08X",$_) } checksum($a) ),"\n";
#}
#exit;

my ($src,$dst,@tables,$prog,$remote,$ssh,@options);

$prog = 'pgrsync';
$ssh = 'ssh';

my %options = ( 'src|s=s'    => \$src, 
                'dest=s'     => \$dst, 
                'tables=s@'  => \@tables, 
                'prog=s'     => \$prog,
                'remote=s'   => \$remote,
                'ssh|e=s'    => \$ssh,
                'options=s@' => \@options );

GetOptions( %options ) || exit;

if( @ARGV )
{
  die "Unrecognised option $ARGV[0]\n";
}

my( $srcdb, $dstdb );

if( defined $remote )
{
  $dstdb = MakeConnection( $remote );
  $srcdb = ConnRemote->newlocal();

  $dstdb->sendinit();
}
else
{
  if( not defined $src or not defined $dst )
  {
    die "Missing required option\n";
  }

  if( @tables == 0 )
  {
    print STDERR "No tables to transfer\n";
    exit;
  }

  @tables = map { split /,/ } @tables;   # Allow comma separated tables as well

  ProcessOptions( @options );

  $srcdb = MakeConnection( $src );
  $dstdb = MakeConnection( $dst );

  my $seed = int( rand(65536) );   # Seed to that coincidences get fix eventually
  $srcdb->settype( 'src', $seed );
  $dstdb->settype( 'dst', $seed );

  if( $srcdb->version ne $dstdb->version )
  {
    die "Sorry, versions do not match [".$srcdb->version." != ".$dstdb->version."]\n";
  }

  $srcdb->settables( @tables );
}

$Data::Dumper::Indent = 0;
while( not $srcdb->{done} and not $dstdb->{done} )
{
  my @msgs = $srcdb->getmessages();
#  print STDERR "SRC => DST: ".Dumper( \@msgs )."\n";
  $dstdb->sendmessages( @msgs );
  
  @msgs = $dstdb->getmessages();
#  print STDERR "DST => SRC: ".Dumper( \@msgs )."\n";
  $srcdb->sendmessages( @msgs );

  microsleep();
}

#if( $src =~ m,^(\w+@)?(\w+)?:(\w+@)?(\w+)(/\d+)?$, )
#my $dba = new PGDb( 'dbname=kleptog' );
#
#my @tables = grep { !/^pg_/ } $dba->getTables;
#
#print "table=",join(",",@tables),"\n";
#
#my $table = $dba->getTable( $tables[0] );
#
#my @attr = $table->getAttributes;
#
#print "attributes=",join(",",@attr),"\n";

sub MakeConnection($)
{
  my $target = shift;

  my($login,$host,$user,$dbname,$port) = ($target =~ m,^(?:(?:(\w+)(?:@))?([\w.-]+))?:(\w+(?:@))?(\w+)(/\d+)?$,);

  if( not defined $dbname )
  {
    die "Could not parse destination '$target'\n";
  }

  my $conn;

  if( defined $host )   # Is remote login
  {
    $conn = new ConnRemote( $login, $host, $user, $dbname, $port );
  }
  else    # Local database
  {
    $conn = new ConnLocal( $user, $dbname, $port );
  }

  return $conn;    
}

sub Decode ($$)
{
  my($def,$str) = @_;

  my @array;
  while( length $def )
  {
    if( $def =~ /^s/ )
    {
      $def = substr($def,1);

      my $offset = index( $str, "\0" );
      if( $offset >= 0 )
      {
        push @array, substr($str, 0, $offset);
        $str = substr($str, $offset+1);
      }
      else
      {
        return undef;
      }
    }
    elsif( $def =~ /^i/ )
    {
      $def = substr($def,1);

      if( length( $str ) >= 4 )
      {
        push @array, unpack("N", substr( $str, 0, 4 ));
        $str = substr( $str, 4 );
      }
      else
      {
        return undef;
      }
    }
    elsif( $def =~ /^c/ )
    {
      $def = substr($def,1);

      if( length( $str ) >= 1 )
      {
        push @array, substr( $str, 0, 1 );
        $str = substr( $str, 1 );
      }
      else
      {
        return undef;
      }
    }
    elsif( $def =~ /^\((.+)\)/ )
    {
      $def = substr($def,length($1)+2);

      my $code = $1;

      if( length($str) < 1 )
      { return undef }

      my $count = ord( substr( $str, 0, 1 ) );
      $str = substr( $str, 1 );

      my @a;
      for(0..$count-1)
      {
        my $arr;
        ($arr,$str) = Decode( $code, $str );

        return undef if not defined $arr;

        push @a, $arr;
      }

      push @array, \@a;
    }
    else
    {
      die "Can't parse format [$def]\n";
    }
  }
  return (\@array,$str);
}

sub GetMessage ($)
{
  my $orig = shift;

  return (undef, $orig) if length($orig) == 0;

  my $type = substr( $orig, 0, 1 );
  my $str = substr( $orig, 1 );

  my $def = $msgtypes{$type};

  if( not defined $def )
  {
    die "Unknown message type [$type] [$str]\n";
  }

  my ($arr,$res) = Decode( $def, $str );

  # not enough for a message
  if( not defined $arr ) { return (undef, $orig) }

  unshift @$arr, $type;

  print STDERR "Received msg: [@$arr]\n";

  return ($arr,$res);
}

sub Encode ($$)
{
  my($def,$arr) = @_;
  $arr = [ @$arr ];
  my $str = "";

  while( length $def )
  {
    die "Message error\n" unless scalar( @$arr );

    if( $def =~ /^s/ )
    {
      $def = substr($def,1);

      $str .= shift(@$arr)."\0";
    }
    elsif( $def =~ /^i/ )
    {
      $def = substr($def,1);

      $str .= pack( "N", shift(@$arr) );
    }
    elsif( $def =~ /^c/ )
    {
      $def = substr($def,1);

      if( length($arr->[0]) != 1 )
      {
        die "Type c wrong length ($arr->[0])\n";
      }
      $str .= shift @$arr;
    }
    elsif( $def =~ /^\((.+)\)/ )
    {
      $def = substr($def,length($1)+2);

      my $code = $1;
      my $list = shift @$arr;

      $str .= chr( scalar( @$list ) );

      for( @$list )
      {
        $str .= Encode( $code, $_ );
      }
    }
    else
    {
      die "Can't parse format [$def]\n";
    }
  }
  die "Too many args\n" if scalar( @$arr );

  return $str;  
}

sub MakeMessage ($)
{
  my $arr = shift;

  $arr = [ @$arr ];    # Copy list

  die unless ref($arr) eq "ARRAY";

  my $type = shift @$arr;

  my $def = $msgtypes{$type};

  if( not defined $def )
  {
    die "Unknown message type [$type,$def]\n";
  }

  my $str = Encode( $def, $arr );

  return $type.$str;
}

package ConnRemote;
use IPC::Open2;
use IO::File;
use IO::Select;

sub new
{
  my $self = bless {}, shift;

  my( $login, $host, $user, $dbname, $port ) = @_;

  my( $rfh, $wfh );
  if( defined($login) and $login ne "" )
  { $host = "$login\@$host" }

  my $t = ":".(defined($user)?"$user@":"").$dbname.(defined($port)?"/$port":"");
  open2( $rfh, $wfh, $ssh, '-C', '-e', 'none', $host, $prog, '-r', $t );

  $self->{wfh} = $wfh;
  $self->{rfh} = $rfh;
  $self->{buffer} = "";
  $self->{done} = 0;
  $rfh->blocking(0);

  return $self;
}

sub newlocal
{
  my $self = bless {}, shift;

  my( $rfh, $wfh );

  $self->{wfh} = new IO::File ">&STDOUT";
  $self->{rfh} = new IO::File "<&STDIN";  
  $self->{buffer} = "";
  $self->{rfh}->blocking(0);

  return $self;
}

sub settype
{
  my $self = shift;
  my $type = shift;
  my $seed = shift;

  my $wfh = $self->{wfh};
  my $rfh = $self->{rfh};

#  $wfh->print( ($type eq "src")?'S':'D' );
#  $wfh->flush();

  my $msg;

  do {
    main::microsleep();
    $msg = $self->getmessage( $rfh );
  } until( defined $msg );

  if( $msg->[0] eq "E" )
  {
    die "Error: $msg->[1]\n";
  }

  if( $msg->[0] eq "V" )
  {
    $self->{versionstring} = $msg->[1];
    my($ver,$db) = ($self->{versionstring} =~ /^pgrsync v(\d+\.\d+) db(\d+\.\d+)\n/);

    if( not defined $ver or not defined $db )
    {
      die "Error in version string: $self->{versionstring}\n";
    }
    if( $ver ne $PROTO )
    {
      die "Protocol mismatch [$ver != $PROTO]\n";
    }
    $self->{version} = $db;
  }
  else
  {
    die "Bad message '$msg->[0]'\n";
  }

  $self->sendmessages( [ 's', ($type eq "src")?'s':'d', $seed ] );
  $self->{type} = $type;

  return $self;
}

sub version
{
  my $self = shift;

  return $self->{version};
}

sub settables
{
  my $self = shift;

  $self->sendmessages( [ "S", [ map { [ $_ ] } @_ ] ] );

  $self->{wfh}->flush();
}

# Reads a message from the queue
sub getmessage
{
  my $self = shift;

  my $data = "";

  my $s = new IO::Select $self->{rfh};

  my @ready = $s->can_read(0);

  if( scalar(@ready) )
  {
    my $len = $self->{rfh}->read( $data, 4096 );   # Is non-blocking

    if( not defined ($len) )   # We've reached the end
    {
      if( length( $self->{buffer} ) == 0 )
      {
        print STDERR "Read returned 0, EOF=".$self->{rfh}->eof."\n";
        $self->{done} = 1;
      }
    }

    if( defined $data )
    {
      $self->{buffer} .= $data;
    }
  }
  my( $arr, $res ) = main::GetMessage( $self->{buffer} );

  if( defined $arr )
  {
    $self->{buffer} = $res;

    if( $arr->[0] eq "Q" )
    { $self->{done} = 1 }

    return $arr;
  }
  return undef;
}

sub getmessages
{
  my $self = shift;

  my @msgs;

  for(;;)
  {
    my $msg = $self->getmessage();

    if( defined $msg )
    { push @msgs, $msg; next }

    last;
  }

  return @msgs;
}

sub sendmessages
{
  my $self = shift;

  my @msgs = @_;

  my $str = "";

  foreach(@msgs)
  {
    if( $_->[0] eq "Q" )
    { $self->{done} = 1 }

    my $s = main::MakeMessage( $_ );

    $str .= $s;
  }

  if( length($str) )
  {
    $self->{wfh}->print( $str );   # Could block I guess
  }
  $self->{wfh}->flush();

  if( $self->{done} )
  { $self->{wfh}->close() }
}

package ConnLocal;
use PGDb;
use Pg;

sub new
{
  my $self = bless {}, shift;

  my ( $user, $dbname, $port ) = @_;

  my $db = new PGDb("dbname=$dbname ".(defined($user)?"user=$user ":"").(defined($port)?"port=$port ":"") );

  $self->{db} = $db;
  $self->{msgqueue} = [];
  $self->{stage} = 0;
  $self->{done} = 0;
  return $self;
}

sub settype
{
  my $self = shift;
  my $type = shift;
  my $seed = shift;

  $self->{type} = $type;
  $self->{seed} = $seed;

  return;
}

sub exec ($)
{
  my $self = shift;

  return $self->{db}->db->exec( shift );
}

sub version
{
  my $self = shift;
  my $res = $self->exec("select version()");

  my $version = $res->getvalue(0,0) || die "Couldn't get version\n";

  my ($val) = ($version =~ /postgresql (\d+\.\d+)\.\d+/i);

  return $val;
}

sub sendinit
{
  my $self = shift;

  my $version = $self->{db}->version;

  $self->queue_response( [ 'V', "pgrsync v$PROTO db$version\n" ] );
}

sub settables
{
  my $self = shift;

  $self->{tables} = [ @_ ];
}

sub getmessages
{
  my $self = shift;

  my $queue = $self->{msgqueue};

  $self->{msgqueue} = [];

  return @$queue;
}

sub getmessage
{
  my $self = shift;

  return @{ $self->{msgqueue} };
}

sub queue_response ($)
{
  my $self = shift;
  push @{ $self->{msgqueue} }, @_;

  return;
}

sub encodestr($)
{
  my $str = shift;

  $str = join( "", map { ($_ eq "\0")?'\0':($_ eq "\\")?'\\':($_ eq "'")?'\\\'':$_ } (split //, $str) );

  return $str;
}

sub fetch_row()
{
  my $self = shift;

  if( not defined $self->{result} )
  {
    $self->{result} = $self->exec( "fetch 1000 from pgrsync_cursor" );
    if( not $self->{pktype} )
    {
      $self->{pktype} = sub { $_[0] <=> $_[1] };

      foreach my $i (0..$self->{result}->ntuples-1)
      {
        my $val = $self->{result}->getvalue( $i, $self->{pkindex} );
        if( $val !~ /^-?\d+(\.\d+)?$/ or $val =~ /^-?0\d/ )   # Not a number
        {
          $self->{pktype} = sub { $_[0] cmp $_[1] };
          last;
        }
      }
    }
    $self->{currrow} = 0;
  }

  if( $self->{currrow} >= $self->{result}->ntuples )
  {
    if( $self->{result}->ntuples == 0 )
    {
      $self->{currentpkey} = undef;
      $self->{result} = undef;
      return;
    }
    $self->{result} = undef;
    $self->fetch_row();
    return;
  }
  else
  {
    my @data = map { 
                     $self->{result}->getisnull( $self->{currrow}, $_ ) ? '\N' : encodestr( $self->{result}->getvalue( $self->{currrow}, $_ ) ) 
                   } (0..$self->{result}->nfields-1);

    $self->{currentpkey} = splice @data, $self->{pkindex}, 1;
    $self->{currdata} = [ @data ];
    $self->{currrow}++;
    return;
  }
}

sub send_update ()
{
  my $self = shift;

  $self->queue_response( [ 'U', $self->{currentpkey}, [ map { [ $_ ] } @{ $self->{currdata} } ] ] );

  return;
}

sub send_insert ()
{
  my $self = shift;

  $self->queue_response( [ 'I', $self->{currentpkey}, [ map { [ $_ ] } @{ $self->{currdata} } ] ] );

  return;
}

sub current_csum ()
{
  my $self = shift;
  return main::checksum( pack( "N", $self->{seed} ).join( "\0", @{ $self->{currdata} } ) );
}

# Process messages
sub sendmessages 
{
  my $self = shift;

  push @{ $self->{incoming} }, @_;

  while( scalar( @{ $self->{incoming} } ) )
  {
    my $msg = shift @{ $self->{incoming} };

    if( $msg->[0] eq "Q" )
    {
      $self->{done} = 1;
      $self->queue_response( [ 'Q' ] );
      next;
    }

    if( not defined $self->{type} )
    {
      if( $msg->[0] eq "s" )    # setup message, initial. Tells the setup
      {
        $self->settype( (($msg->[1] eq 's')?'src':'dst'), $msg->[2] );  # Set type and seed
      }
    }
    elsif( $self->{type} eq "src" )   # Source of data
    {
      if( $msg->[0] eq "S" )    # Setup the table list to send
      {
        $self->settables( map { $_->[0] } @{ $msg->[1] } );
      }
      elsif( $msg->[0] eq "B" )   # Beginning to get table data
      {
      # Handle B: Set table. Open cursor. Load data for buffer.
        my $tableinfo = $self->{tabledefs}->{$msg->[1]};

        if( not defined $tableinfo )
        {
          print STDERR "Huh? B message for table I don't know about '$msg->[1]'\n";
          next;
        }

        $self->exec( "begin" );            # Use query given by MapTable
        my $cursor = $self->exec( "declare pgrsync_cursor cursor for ". $tableinfo->[1] );
        if( $cursor->resultStatus != PGRES_COMMAND_OK )
        {
          $self->queue_response( [ 'A' ] );   # Abort, If you don't do this, the destination may delete the whole table :(
          $self->queue_response( [ 'E', 'Query error: '.$self->{db}->db->errorMessage ] );
          $self->{stage} = 3;     # Aborted state
          $self->exec( "abort" );
          return;
        }
        $self->{stage} = 2;
        $self->{pkindex} = $tableinfo->[0]->[2];  # So fetch_row knows which it is

        $self->{pktype} = undef;    # Reset numeric/text test
        $self->fetch_row();
      }
      elsif( $msg->[0] eq "C" )
      {
        if( $self->{stage} == 3 )    # We've aborted. Ignore these till the F comes through
        { next }
      # Handle C: Check if pkey matches next row in buffer
      #     If less, send Delete message, next
      #     If equal, check checksum, send U on mismatch, next
      #     If greater, send U for Insert for each row until less-or-equal
        for(;;)
        {
          if( not defined $self->{currentpkey} )   # We've reached end of input, delete remainder
          {
            $self->queue_response( [ 'D', $msg->[1] ] );
            last;
          }
          my $cmp = $self->{pktype}->( $msg->[1], $self->{currentpkey} );

          if( $cmp < 0 )  # We don't have it, they do
          {
            $self->queue_response( [ 'D', $msg->[1] ] );
            last;
          }
          elsif( $cmp == 0 )  # Same pkey, now for checksums
          {
            if( $msg->[2] != $self->current_csum() )    # Checksum mismatch, send update
            {
              $self->send_update();
            }
            $self->fetch_row();       # Onto next row
            last;
          }
          elsif( $cmp > 0 )   # We have it, other end doesn't
          {
            $self->send_insert();     # Send insert data
            $self->fetch_row();       # Get next row
            next;                     # And loop again
          }
        }
      }
      elsif( $msg->[0] eq "F" )
      {
      # Handle F: Send I records for any remaining data
        if( $self->{stage} != 3 )   # If we're not aborted
        {
          while( defined $self->{currentpkey} )
          {
            $self->send_insert();     # Send insert data
            $self->fetch_row();       # Get next row
          }
          $self->queue_response( [ "F" ] );  # Acknowledge completion
          $self->exec( "commit" );
        }

        $self->{stage} = 1;
      }
      elsif( $msg->[0] eq "A" )    # Other end aborted
      {
        if( $self->{stage} != 3 )   # If we're not aborted already
        {
          $self->queue_response( [ "F" ] );  # Finish abort
          $self->exec( "abort" );
        }

        $self->{stage} = 1;
      }
      elsif( $msg->[0] eq "E" )
      {
      # Handle E: Display error message
        print STDERR "ERROR: $msg->[1]\n";
      }
      else
      {
        print STDERR "Unknown message: ".Dumper( $msg );
      }
    }
    elsif( $self->{type} eq "dst" )    # Destination of data
    {
      if( $msg->[0] eq "T" )   # Receive table def
      {
        if( not defined $self->{tabledefs} ) { $self->{tabledefs} = [] }
        push @{ $self->{tabledefs} }, [ @$msg ];   # Make copy of message
        next;
      }
      # Handle F: Mark this table as done
      elsif( $msg->[0] eq "F" )   # Other end has completed table, we reset to begin next table
      {
        if( $self->{stage} != 4 )   # Not aborted?
        {
          print STDERR "Table $self->{currtabledef}->[1]: +$self->{stats}->{ins} -$self->{stats}->{del} #$self->{stats}->{upd}\n";
          $self->exec("commit");
        }
        else
        {
          $self->exec("abort");
        }
        $self->{stage} = 0;
        next;
      }
      my $tablename = $self->{currtabledef}->[1];
      my $tableattrs = $self->{currtabledef}->[3];
      my $pkindex = $self->{pkindex};
      my $pkeyname = $tableattrs->[ $pkindex ]->[0];

      # Handle U: Check we are in update mode. If so, execute update
      if( $msg->[0] eq "U" )
      {
        next if $self->{stage} == 4;    # If we've aborted, ignore

#        print STDERR "Update pkey $msg->[1]\n";
        my $query = "update \"$tablename\" set ".
                        join(",", map { "\"$tableattrs->[$_+($_ >= $pkindex)]->[0]\" = ".
                                        (($msg->[2]->[$_]->[0] eq '\N')?'null':"'$msg->[2]->[$_]->[0]'") } 
                                 (0..scalar(@$tableattrs)-2)).
                    " where \"$pkeyname\" = '$msg->[1]';\n";
        my $res = $self->exec( $query );
        if( $res->cmdTuples ne 1 )
        { $self->queue_response( [ 'E', $self->{db}->errorMessage ] ); 
          if( $self->{stage} == 2 ) 
          { 
            $self->queue_response( [ 'A' ] ); 
          } 
          $self->{stage} = 4;
        }
        $self->{stats}->{upd} ++;
      }
      elsif( $msg->[0] eq "I" )
      {
        next if $self->{stage} == 4;    # If we've aborted, ignore

#        print STDERR "Insert pkey $msg->[1]\n";
        my $query = "insert into \"$tablename\" (\"$pkeyname\",".
                        join(",", map { "\"$tableattrs->[$_]->[0]\"" } (0..($pkindex-1),($pkindex+1)..scalar(@$tableattrs)-1)).
                    ") values ('$msg->[1]',".
                        join(",", map { ($_->[0] eq '\N')?'null':"'$_->[0]'" } @{ $msg->[2] }).");\n";
        my $res = $self->exec( $query );
        if( $res->cmdTuples ne 1 )
        { $self->queue_response( [ 'E', $self->{db}->errorMessage ] ); 
          if( $self->{stage} == 2 ) 
          { 
            $self->queue_response( [ 'A' ] ); 
          } 
          $self->{stage} = 4;
        }
        $self->{stats}->{ins} ++;
      }
      # Handle D: If not in preserve mode, execute delete
      elsif( $msg->[0] eq "D" )
      {
        next if $self->{stage} == 4;    # If we've aborted, ignore

#        print STDERR "Delete pkey $msg->[1]\n";
        my $query = "delete from \"$tablename\" where \"$pkeyname\" = '$msg->[1]';\n";
        my $res = $self->exec( $query );
        if( $res->cmdTuples ne 1 )
        { $self->queue_response( [ 'E', $self->{db}->errorMessage ] ); 
          if( $self->{stage} == 2 ) 
          { 
            $self->queue_response( [ 'A' ] ); 
          } 
          $self->{stage} = 4;
        }
        $self->{stats}->{del} ++;
      }
      elsif( $msg->[0] eq "A" )    # Source has error, abort
      {
        $self->exec( "abort" );
        if( $self->{stage} != 4 )   # Only if we havn't aborted already
        {
          print STDERR "Source aborted transfer of table\n";
          if( $self->{stage} == 2 )      # If we're still sending data, send finish message
          {
            $self->queue_response( [ 'F' ] ); 
          }
        }
        $self->{stage} = 0;   # Back to normal
      }
      elsif( $msg->[0] eq "E" )    # Source sent error message
      {
        print STDERR "ERROR: $msg->[1]\n";
      }
      else
      {
        print STDERR "ERROR: Unknown type $msg->[0]\n";
      }
    }
  }

  return unless defined $self->{type};   # Can't do anything unless we know our type

  if( $self->{type} eq "src" )
  {
    # Stage 0: init. When we have the table list, work out their structure and send it
    # Stage 1: Waiting for remote to begin sending table
    # Stage 2: Remote has started sending checksums
    # Stage 3: Transfer has aborted, wait for crap to clear

    if( $self->{stage} == 0 )
    {
      return unless $self->{tables};

      if( not defined $self->{tabledefs} ) { $self->{tabledefs} = {} }
      foreach my $tablename ( @{ $self->{tables} } )
      {
        my $table = main::GetTableDef( $self->{db}, $tablename );
        if( $table->[0] eq "T" )    # Return table def
        {
          my( $dsttabledef, $query ) = main::MapTable( $table );   # Do table mapping. After this we only use the new name.

          $self->queue_response( $dsttabledef );
          $self->{tabledefs}->{$dsttabledef->[1]} = [ $dsttabledef, $query ];    # Put table data within hash
        }
        else
        {
          $self->queue_response( $table );   # Send error
        }
      }
      $self->{stage} = 1;   # Wait for remote
    }
  }
  else   # We're the destination
  {
    # Stage 0: init. Waiting for destination to send table destinations
    # Stage 1: We've got a table, we're starting to send table
    # Stage 2: We're sending data
    # Stage 3: All data has been sent, wait for other end to complete
    # Stage 4: We've aborted the transfer, waiting for F
    if( $self->{stage} == 0 )
    {
      return unless $self->{tabledefs};   # Gotta have a table to work with :)
      if( scalar( @{ $self->{tabledefs} } == 0 ) )   # No tables left, must be done
      {
        $self->queue_response( [ 'Q' ] );
        $self->{done} = 1;
        return;
      }

      $self->{stage} = 1;
    }
    if( $self->{stage} == 1 )     # Initial table
    {
      my $table = shift @{ $self->{tabledefs} };

      # Here we need to check that the format in our DB matches the format
      # given. If error, return E and go for next. Else we send a begin
      # record and open a cursor on the table. Set the stage to 2 so we send
      # data.

      my $ourtable = main::GetTableDef( $self->{db}, $table->[1] );

      if( $ourtable->[0] eq "E" )
      {
        $self->queue_response( [ 'E', "Destination: $table->[1]: Cannot copy, please create first" ] );
        $self->{stage} = 0;
        return;
      }

      # Check name of primary key
      if( $table->[3]->[$table->[2]]->[0] ne $ourtable->[3]->[$ourtable->[2]]->[0] )
      {
        $self->queue_response( [ 'E', "Table $table->[1]: Primary key mismatch" ] );
        $self->{stage} = 0; 
        return;
      }

      ATTR: foreach my $attr ( @{ $table->[3] } )
      {
        my ($name,$type) = @$attr;

        foreach my $a ( @{ $ourtable->[3] } )
        { 
          if( $a->[0] eq $attr->[0] )
          {
# We no longer bother with attribute types. User problem.
#            if( $a->[1] == $attr->[1] )
#            {
              next ATTR;
#            }
#            $self->queue_response( [ 'E', "Table $table->[1]: Attribute $a->[0]: type mismatch" ] );
#            $self->{stage} = 0; 
#            return;
          }
        }
        $self->queue_response( [ 'E', "Table $table->[1]: Attribute $attr->[0]: field not found" ] );
        $self->{stage} = 0; 
        return;
      }
      $self->exec( "begin" );
      my $cursor = $self->exec( "declare pgrsync_cursor cursor for select ".
                   join( ", ", map { "\"$_->[0]\"" } @{ $table->[3] } ).
                   "from \"$table->[1]\" ".
                   "order by \"$table->[3]->[$table->[2]]->[0]\"" );
      if( $cursor->resultStatus != PGRES_COMMAND_OK )
      {
        $self->queue_response( [ 'E', 'Query error: '.$self->{db}->db->errorMessage ] );
        $self->{stage} = 0; 
        $self->exec( "abort" );
        return;
      }
      $self->{pkindex} = $table->[2];
      $self->{currtabledef} = $table;
      $self->{stats} = { ins => 0, del => 0, upd => 0 };
      $self->queue_response( [ 'B', $table->[1] ] );
      $self->{stage} = 2;  # Next stage
    }
    if( $self->{stage} == 2 )     # Sending data
    {
      # Fetch 100 rows from the table. Calculate the checksum for each row.
      # Send it with the primary key. If we don't get any rows, close the
      # cursor and shift back to stage 1. Send a finish record.

      for (1..100)
      {
        $self->fetch_row();

        if( not defined $self->{currentpkey} )
        {
          $self->queue_response( [ 'F' ] );   # Note: We cannot commit here, wait for other end to complete
          $self->{stage} = 3;
          return;
        }

        my $csum = $self->current_csum();

        my $key = $self->{currentpkey};

        $self->queue_response( [ 'C', $key, $csum ] );
      }
    }
    # Stage 3: We do nothing until other end sends "F", pushing us back to state 0
  }
}

package main;

my @table;

# This is ofcourse *totally* bogus, but it's non-linear, which CRC-32 isn't
sub inittable ()
{
  #              0   1   2   3    4   5   6   7   8   9  10  11  12  13  14  15
  my @seeds = (  2,  3,  5,  7,  11, 13,  1,  4,  9, 10, 14,  8,  0, 15, 12,  6 );

  # 0 => 2 => 5 => 13 => 15 => 6 => 1 => 3 => 7 => 4 => 11 => 8 => 9 => 10 => 14 => 12 => 0

  for my $i (0..255)
  {
    my $a = $i >> 4;
    my $b = $i & 15;
    my $c = $seeds[ ($a + $b) & 15 ];
    my $d = $seeds[ ($c + $a) & 15 ];
       $c = $seeds[ ($d + $c) & 15 ];
       $d = $seeds[ ($b + $d) & 15 ];

#    print "$i = ($a,$b,$c,$d)\n";

    $table[$i] = $c * 16 + $d;
  }
#  foreach my $a (0..15)
#  {
#    foreach my $b (0..15)
#    {
#      printf "%02X ", $table[ $a * 16 + $b ];
#    }
#    print "\n";
#  }
}

# Once again, completely 100% bogus, but it serves fine
sub checksum ($)
{
  my $str = shift;

  inittable() unless defined $table[0];

  my @arr = unpack "C*", $str;

  my @boxes = ( 0x23, 0x57, 0xDE, 0xA8 );

  foreach my $n (@arr)
  {
    $boxes[0] ^= $table[ $boxes[1] ^ $n ];
    $boxes[1] ^= $table[ $boxes[2] ^ $n ];
    $boxes[2] ^= $table[ $boxes[3] ^ $n ];
    $boxes[3] ^= $table[ $boxes[0] ^ $n ];
  }

  return unpack( "N", pack( "CCCC", @boxes ) );
}

sub microsleep
{
  select undef, undef, undef, 0.1;
}

sub GetTableDef ($)
{
  my($db,$table) = @_;

  my $t = $db->getTable( $table );

  if( not defined $t )
  {
    return [ 'E', "Table $table not found" ];
  }

  my @attrs = $t->getAttributes;    # Get list of attributes

  my @list;

  foreach my $a (@attrs)
  {
    # We don't get the attribute type anymore. Too hard for the time being
#    my $attr = $t->getAttribute( $a );
#
#    my $typeoid = $attr->{typeoid};
#
#    push @list, [ $a, $typeoid ];

    push @list, [ $a ];
  }

  my $pkindex = $t->getPkeyIndex();

  if( not defined $pkindex )
  {
    return [ 'E', "No suitable index found for table $table" ];
  }
  my $descr = [ 'T', $table, $pkindex, \@list ];

#  print STDERR Dumper($descr),"\n";

  return $descr;
}

my %mappings;

# Parse -o options given on command line
sub ProcessOptions
{
  my @options = @_;

  foreach my $opt (@options)
  {
    if( $opt =~ /^map:(\w+)=(\w+)$/ )    # Table name mapping
    {
      $mappings{tables}{$1} = $2;
    }
    elsif( $opt =~ /^map:(\w+)\.(\w+)=(\w+)$/ )    # Table field name mapping
    {
      $mappings{tablefields}{$1}{$2} = $3;
    }
    elsif( $opt =~ /^expr:(\w+)\.(\w+)=(.*)$/ )    # Table field expressions
    {
      $mappings{exprs}{$1}{$2} = $3;
    }
    elsif( $opt =~ /^cond:(\w+)=(.*)$/ )           # Table conditions
    {
      if( not defined $mappings{conds}{$1} )
      { $mappings{conds}{$1} = [] }

      push @{ $mappings{conds}{$1} }, $2;
    }
    elsif( $opt =~ /^fields:(\w+)=(.*)$/ )         # Table field list
    {
      my @list = split /,/,$2;
      if( scalar( grep { not /^!?(\*|\w+)$/ } @list ) )
      {
        die "Couldn't process field list\n";
      }
      $mappings{fields}{$1} = [ @list ];
    }
    else
    {
      die "Couldn't parse option '$opt'\n";
    }
  }
}

# Takes the table as given by GetTableDef and returns two structures. One
# which is the table structure to send to the other end, the other is the
# query to execute at the local end.
sub MapTable
{
  my( $tabledef ) = shift;
  # Tabledef = [ 'T', $tablename, $pkindex, [ [ $fieldname1, $fieldtype1 ], ... ] ]

  my $tablename = $tabledef->[1];
  my( @fieldexprs, @fieldnames, @newfieldnames, $pkindex );

  @fieldnames = map { $_->[0] } @{ $tabledef->[3] };
  @fieldexprs = map { "\"$_\"" } @fieldnames;           # Surround by double quotes
  $pkindex = $tabledef->[2];

  # Step 1: Handle expr: options
  if( defined $mappings{exprs} and defined $mappings{exprs}{$tablename} )
  {
    my $exprlist = $mappings{exprs}{$tablename};

    EXPR: foreach my $field ( keys %$exprlist )
    {
      if( $field eq $fieldnames[ $pkindex ] )
      {
        die "Table $tablename: Field $field: You cannot replace the primary key with an expression\n";
      }
      foreach my $i (0..$#fieldnames)
      {
        if( $fieldnames[$i] eq $field )   # Replacing existing field
        {
          $fieldexprs[$i] = $exprlist->{$field};
          next EXPR;
        }
      }
      # Not found, so new field
      push @fieldnames, $field;
      push @fieldexprs, $exprlist->{$field};
    }
  }
  # Step 2: Handle map:table.field= options
  @newfieldnames = @fieldnames;
  if( defined $mappings{tablefields} and defined $mappings{tablefields}{$tablename} )
  {
    my $maplist = $mappings{tablefields}{$tablename};

    MAP: foreach my $field ( keys %$maplist )
    {
      foreach my $i (0..$#fieldnames)
      {
        if( $fieldnames[$i] eq $field )   # Field found?
        {
          $newfieldnames[$i] = $maplist->{$field};
          next MAP;
        }
      }  
      die "Couldn't find field '$field' for mapping\n";
    }
    # Lame but effective way to test for duplicates
    my %hash = map { ($_ => 0) } @newfieldnames;
    if( scalar(keys %hash) != scalar(@newfieldnames) )
    {
      die "Table $tablename: Mapped fields have duplicate names\n";
    }
  }
  # Step 3: Handle fields: options
  if( defined $mappings{fields} and defined $mappings{fields}{$tablename} )
  {
    my $fieldlist = $mappings{fields}{$tablename};

    my %hash = map { ($_ => 0) } @newfieldnames;   # Default is no

    print STDERR "Fieldlist = ".Dumper($fieldlist)."\n";

    foreach my $descr (@$fieldlist)
    {
      my $value = 1;
      my $field;
      if( $descr =~ /^!/ )   # Has a not in front
      {
        $value = 0;
        $field = substr( $descr, 1 );
      }
      else
      {
        $field = $descr;
      }
    
      if( $field eq "*" )
      {
        foreach my $key (keys %hash)
        { $hash{$key} = $value }
      }
      elsif( defined $hash{$field} )
      {
        $hash{$field} = $value;
      }
      else
      {
        die "Could find $field when processing fieldlist\n";
      }
    }
    if( $hash{ $newfieldnames[ $pkindex ] } == 0 )
    {
      die "Table $tablename: Must include primary key in output list\n";
    }
    my ( @fieldexprs2, @newfieldnames2, $pkindex2 );
    print STDERR "X ".Dumper( \%hash )."\n";
    foreach my $i (0..$#newfieldnames)
    {
      if( $hash{ $newfieldnames[$i] } == 1 )     # Is include
      {
        push @newfieldnames2, $newfieldnames[$i];
        push @fieldexprs2,    $fieldexprs[$i];
        if( $i == $pkindex ) { $pkindex2 = $#newfieldnames2 }
      }
    }
    @fieldexprs = @fieldexprs2;
    @newfieldnames = @newfieldnames2;
    $pkindex = $pkindex2;
  }

  # Stop 4: Map table name
  if( defined $mappings{tables} and defined $mappings{tables}{$tablename} )
  {
    $tablename = $mappings{tables}{$tablename};
  }

  my $returntabledef = [ 'T', $tablename, $pkindex, [ map { [ $_ ] } @newfieldnames ] ];

  my $query = "SELECT ".join(", ", @fieldexprs)." FROM \"$tablename\"";

  if( defined $mappings{conds} and defined $mappings{conds}{$tablename} )
  {
    my $conditionlist = $mappings{conds}{$tablename};

    $query .= " WHERE ".join( " AND ", map { "($_)" } @$conditionlist );
  }
  $query .= " ORDER BY ".($pkindex+1);

  print STDERR "Input: ".Dumper( $tabledef )."\n";
  print STDERR "Output: ".Dumper( $returntabledef )."\n";
  print STDERR "Query: $query\n";

  return ($returntabledef, $query);
}
