/********************************************
 Plan for qipxclient
 
 On startup
 1. Broadcast for IPX servers
 2. Give user choice of server
 
 For the proxy
 3. Open an UDP:27000 socket
 4. Read from socket (will block)
 5. Make sure return address is ourselves
 6. Check that it is a signon message
 7. Forward the signon message
 8. In reply message, fudge the socket number to our socket
 9. Fork and handle messages in both directions

*******************************************/

#include <stdio.h>        // General libraries
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <sys/time.h>
#include <unistd.h>
#include <signal.h>
#include <sys/wait.h>

#include <sys/socket.h>   // General socket stuff
#include <netdb.h>

#include <arpa/inet.h>    // TCP/IP   (UDP)
#include <linux/ipx.h>    // IPX

#include "qipx.h"

#define MAX_CONNECTIONS      32     // Maximum expected servers on the current network
#define BUFFER_LENGTH      2048     // No message should be bigger than this!!!

char buffer[BUFFER_LENGTH];         // Static buffer for storing messages

char *ProgramName;
char RealProgramName[] = "qipxclient";

int otherpid;         // After fork, what is the pid of the other

char msgQuakeBroadcast[] = { 0x00, 0x00, 0x00, 0x00,   // 4 extra bytes for IPX !?
                             0x80, 0x00, 0x00, 0x0C,
                             0x02, 'Q' , 'U' , 'A' ,
                             'K' , 'E' , 0x00, 0x03 };
                             
#define lenQuakeBroadcast sizeof(msgQuakeBroadcast)

FILE *messagefile = stderr;

// They can be the same because serverport is IPX and other is UDP
int portnum = 26000;  // Port number to use
int serverport = 26000;
int messagelevel = 1; // messagelevel
char serveraddress[6] = { 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF };

void HandleCommandline(int argc, char *argv[])
{
  int opt; 
  
  ProgramName = argv[0];
  
  while( (opt=getopt(argc,argv,"p:m:s:a:o:h")) != EOF)
  {
    switch(opt)
    {
      case '?':
      case 'h':
        printf("qipxclient v1.0 - allow UDP clients to connect to an IPX server\n"
               "Copyright Oct 1997 by Martijn van Oosterhout. See copyright file for details\n"
               "qipxclient [-p port] [-s port] [-a addr] [-o file] [-m messagelevel] [-h]\n"
               "  -p port          Changes port to react to (default 26000)\n"
               "  -s port          Port the Quake server is on (default 26000)\n"
               "  -a xxxxxxxxxxxx  Specifiy ethernet address of server\n"
               "  -o file          Specify file to log output to (default stdout)\n"
               "  -m messagelevel  Changes verbosity of messages (default 1)\n"
               "                   0. No messages\n"
               "                   1. Status messages\n"
               "                   2. Displaying of network addresses\n"
               "                   3. Dumping of negotiation packets\n"
               "                   4. Dumping of all packets (_LOTS_ of data!!)\n");
        exit(0);
      case 'p':
        portnum = atoi(optarg);
        if(portnum == 0)
        {
          printf("qipxclient: Illegal port number\n");
          exit(1);
        }
        break;
      case 's':
        serverport = atoi(optarg);
        if(serverport == 0)
        {
          printf("qipxclient: Illegal port number\n");
          exit(1);
        }
        break;
      case 'm':
        messagelevel = atoi(optarg);
        break;
      case 'a':
        if(ParseIPXaddress(serveraddress,optarg))
          exit(1);
        break;
      case 'o':
        messagefile = fopen(optarg,"wt");
        
        if(!messagefile)
        {
          fprintf(stderr,"qipxclient: Can't open output file '%s' (%s)\n",optarg,strerror(errno));
          exit(0);
        }
        break;
    }
  }
}

#if 0
void SendKeepAlive( int ipxsock, struct sockaddr_ipx *addr)
{
  static char KeepAlive[] = { 0x00, 0x00, 0x00, 0x00,
                              0x00, 0x10, 0x00, 0x09,
                              0x00, 0x00, 0x00, 0x00,
                              0x01 };
                              
  int length;
           
  message(2,"qipxclient: Send keepalive message\n");
  messagedump(4,KeepAlive,sizeof(KeepAlive));
                     
  Check( sendto( ipxsock, KeepAlive, sizeof(KeepAlive), 0, (struct sockaddr*)addr, sizeof( *addr ) ), "Sendto" );
  
  alarm(5);
  
  Check( length = recv( ipxsock, buffer, BUFFER_LENGTH, 0 ), "recv" );
  
  message(2,"qipxclient: Keepalive response\n");
  messagedump(4, buffer, length);
}
#endif

void GetServer(int ipxsock, struct sockaddr_ipx *addr)
{
  struct sockaddr_ipx sources[MAX_CONNECTIONS];
  int numsources = 0;
  int length,addrlen,choice;
  
  struct sockaddr_ipx destaddr;
  
  addrlen = sizeof(struct sockaddr_ipx);
  destaddr.sipx_family = AF_IPX;               // Setup broadcast IPX:26000
  destaddr.sipx_port = htons(serverport);
  destaddr.sipx_network = 0;
  destaddr.sipx_type = 17;
  memcpy(&destaddr.sipx_node[0],serveraddress,sizeof(destaddr.sipx_node));
  
  message(2,"qipxclient: Broadcast address is %s\n",PrintIPXaddress(&destaddr));
  
  message(1,"qipxclient: Broadcasting for Quake servers\n");
  messagedump(3,msgQuakeBroadcast,lenQuakeBroadcast);
  // Send broadcast to find servers
  sendto(ipxsock,msgQuakeBroadcast,lenQuakeBroadcast,0,(struct sockaddr*)&destaddr,sizeof(destaddr));
  
  while(numsources < MAX_CONNECTIONS)
  {
    int result;
  
    fd_set set;
  
    struct timeval tv;
  
    tv.tv_sec = 2;
    tv.tv_usec = 0;
  
    FD_ZERO(&set);
  
    FD_SET(ipxsock,&set);
  
    // Wait for two seconds for a response
    Check(result = select(ipxsock+1,&set,NULL,NULL,&tv),"Select");
  
    if(!result)
      break;
     
    if(FD_ISSET(ipxsock,&set))
    {
      addrlen = sizeof(struct sockaddr_ipx);
      Check(length = recvfrom(ipxsock,buffer,BUFFER_LENGTH,0,(struct sockaddr*)&sources[numsources],&addrlen),"recvfrom");
      
      if((length-4) != htons(*(short*)(buffer+6)))
      {
        message(1,"qipxclient: Packet size error (%d != %d)\n", length-4, htons(*(short*)(buffer+6)));
      }
      
      message(2,"qipxclient: Got reply message\n");
      messagedump(3,buffer,length);

      if(length > 10 && buffer[8] == 0x83)
      {
        char *str = &buffer[9];
        
        fprintf(stderr,"%d. %s ",numsources+1,str);
        str += strlen(str)+1;

        fprintf(stderr,"%s ",str);
        str += strlen(str)+1;

        fprintf(stderr,"%s ",str);
        str += strlen(str)+1;
        
        fprintf(stderr,"%d/%d\n",str[0],str[1]);
        
        numsources++;
      }
    }
  }
  
  if(numsources == 0)
  {
    fprintf(stderr,"No servers found. Quitting\n");
    exit(0);
  }
  
  if(numsources == MAX_CONNECTIONS)
  {
    message(1,"qipxclient: Warning: More than %d servers, not all listed\n",MAX_CONNECTIONS);
  }

  if(numsources !=  1)
  {  
    fprintf(stderr,"\nMake a choice (blank to quit): ");
    fflush(stdout);
  
    fgets(buffer,10,stdin);
  
    choice = atoi(buffer);
  
    if( choice < 1 || choice > numsources)
    {
      fprintf(stderr,"qipxclient: Quitting\n");
      exit(0);
    }
  }
  else
  {
    message(1,"qipxclient: Automatically choosing only one\n");
    choice = 1;
  }
  
  memcpy(addr,&sources[choice-1],sizeof(struct sockaddr_ipx));
  
  message(2,"qipxclient: Chose server %s\n",PrintIPXaddress(addr));
  
#if 0  
  SendKeepAlive(ipxsock, addr);
#endif
}

// To kill off friend if something goes wrong
void KillBuddy(int status)
{
  kill(otherpid,SIGTERM);
  if( status == SIGALRM )
    message(0,"Quitting due to timeout\n");
  else
    message(0,"Quitting due to signal %d\n", status);
  exit(0);
}

void ChildDeathHandler(int status)      // Fudge unused parameter
{
  wait(&status); // Clean up child    // Use rubbish parameter
}

void main(int argc, char* argv[])
{
  int udp26000sock;            // Handle of socket UDP:26000
  int udpsock;                 // Handle of other UDP socket
  int ipxsock;                 // Handle of IPX socket
  
  int length;                  // Length of the read data
  
  struct sockaddr_in addr;     // For receiving the address of the client
  int addrlen;                 // The length of the IPX in readfrom
  
  struct sockaddr_ipx destaddr;// Broadcast address IPX:26000
  
  int count = 0;               // Count of messages used by ipx
  int localserversocket;       // Local udpsocket
  
  int pid;
  
  HandleCommandline(argc,argv);
  
  ipxsock = openipxsocket(0);         // Open the IPX socket
  
  message(1,"qipxclient: Checking for servers\n");
  
  GetServer(ipxsock,&destaddr);  // Get replies and ask user
  
  // destaddr now contains address of server to send connect to.
  
  udp26000sock = openudpsocket(portnum);  // Open server socket
  udpsock = openudpsocket(0);           // Open communication socket
  
  addrlen = sizeof(addr);
  Check(getsockname(udpsock,(struct sockaddr*)&addr,&addrlen),"Getsockname");
  localserversocket = htons(addr.sin_port);
  
  message(2,"qipxclient: Local server socket is %d\n",localserversocket);
  
  mysignal( SIGCHLD, ChildDeathHandler );

  for(;;) // Until connection is made
  {
    for(;;)
    {
      message(1,"qipxclient: Waiting for client...\n");
      
      // Now wait for client to join.
      addrlen = sizeof(addr);
      Check(length = recvfrom(udp26000sock,buffer+4,BUFFER_LENGTH-4,0,(struct sockaddr*)&addr,&addrlen), "recvfrom");   // Wait for a packet
      // addr now contains address of client
    
      *(long*)buffer = count++;

      if(length != 12 || buffer[8] != 0x01)
      {
        message(1,"qipxclient: Not a connect message\n");
        messagedump(3,buffer,length+4);
        
        Check( sendto(ipxsock, buffer, length+4, 0, (struct sockaddr*)&destaddr, sizeof(destaddr) ), "Sendto" );
        message(2,"qipxclient: Forwarding message\n");
        
        if( !fork() )
        {
          alarm(10);
          
          Check( length = recv( ipxsock, buffer, BUFFER_LENGTH, 0 ), "recv" );
          message(2,"qipxclient: Received reply. Forwarding...\n");
          messagedump(4,buffer,length);
          
          Check( sendto( udp26000sock, buffer+4, length-4, 0, (struct sockaddr*)&addr, sizeof(addr) ), "sendto" );
          
          exit(0);
        }
        
        continue;
      }
      
      message(1,"qipxclient: Received connect message\n");
      messagedump(3,buffer,length+4);
      break;
    }
    
    message(2,"Client address : %s\n",PrintUDPaddress(&addr));
    message(2,"Server address : %s\n",PrintIPXaddress(&destaddr));
    
    // Forward connect message
    Check(sendto(ipxsock,buffer,length+4,0,(struct sockaddr*)&destaddr,sizeof(destaddr)),"Sendto"); 
    
    message(1,"qipxclient: Waiting for server reply\n");
  
    // Receive server message
    Check(length = recv(ipxsock,buffer,BUFFER_LENGTH,0),"recv");
  
    if(buffer[8] == 0x81)
    {
      message(1,"qipxclient: Connection accepted\n");
      messagedump(3,buffer,length);
      
      // Connection accepted
      // Set destaddr to point to server communication address
      destaddr.sipx_port = htons(*(unsigned short*)(buffer+9));
      
      message(2,"qipxclient: New server address : %s\n",PrintIPXaddress(&destaddr));
      
      // Tell client to use the local server port
      *(short*)(buffer+9) = localserversocket;
    
      message(3,"qipxclient: Forwarding modified message\n");
      messagedump(3,buffer,length);
      
      // Forward message to client
      Check(sendto(udp26000sock,buffer+4,length-4,0,(struct sockaddr*)&addr,sizeof(addr)),"Sendto");
    
      // Negotiation complete, start proxy
      break;
    }
  
    message(1,"qipxclient: Connection refused\n");
    messagedump(3,buffer,length);
    // Connection refused. Forward message unmodified
    Check(sendto(udp26000sock,buffer+4,length-4,0,(struct sockaddr*)&addr,sizeof(addr)),"Sendto");
  }
  
  // Starting proxy
  message(1,"qipxclient: Starting proxy\n");
  
  close(udp26000sock);
  
  mysignal(SIGPIPE,KillBuddy);
  mysignal(SIGALRM,KillBuddy);
  alarm(60);
    
  Check(pid = fork(), "Fork");
  
  if(pid) // Parent, udp->ipx
  {
    SetProgramName("qipxclient: UDP(%s) -> IPX(%s)",PrintUDPaddress(&addr),PrintIPXaddress(&destaddr));
    otherpid = pid;
    
    for(;;)
    {
      Check(length = recv(udpsock,buffer+4,BUFFER_LENGTH-4,0),"Parent recv");  
      *(long*)buffer = count++;
      Check(sendto(ipxsock,buffer,length+4,0,(struct sockaddr*)&destaddr,sizeof(destaddr)),"Parent sendto");
      message(4,"qipxclient: Client->Server\n");
      messagedump(4,buffer,length+4);
      
      if( *(long*)(buffer+4) == htonl( 0x00100009 ) && buffer[12] == 0x02)  // Disconnect packet
      {
        message(1,"qipxclient: Client quitting\n");
        
        kill( otherpid, SIGTERM );
        exit(0);
      }
      
      alarm(240);
    }
  }
  else // Child, ipx->udp
  {
    SetProgramName("qipxclient: IPX(%s) -> UDP(%s)",PrintIPXaddress(&destaddr),PrintUDPaddress(&addr));
    otherpid = getppid();
    
    for(;;)
    {
      Check(length = recv(ipxsock,buffer,BUFFER_LENGTH,0),"Child recv");
      Check(sendto(udpsock,buffer+4,length-4,0,(struct sockaddr*)&addr,sizeof(addr)),"Child sendto");
      message(4,"qipxclient: Server->Client\n");
      messagedump(4,buffer,length);
      
      alarm(240);
    }
  }
}
