/********************************************
 Quake server login sequence
 
 1. Send key string to port 26000. The string is 80 00 00 0C 01 51 55 41 4B 45 00 03.
 2. Server sends from 26000 the message 80 00 00 09 81 xx xx 00 00 where xxxx is the
    port to use.
 3. Server then sends a message from that port describing the level, models required
    etc. When this is received, the client says connection accepted.
    
 Quake then sends messages from that port. What else happens, I don't know
********************************************/
/********************************************
 Plan for qipxserver
 
 1. Open socket IPX:26000
 2. Read from socket (will block)
 3. Check return address whether we've seen this person before?
 4. If so, send packet to UDP:26000. Goto step 2
 5. Otherwise, open a UDP socket, spawn new child to handle 
        the other socketsand send packet to UDP:26000. Goto step 2
        
 Steps for child processes
 6. Read from new UDP socket (will block)
 7. Extract port address from reply
 8. Send via IPX:26000 to appropriate person
 9. Open new IPX socket for the new port address
 
*******************************************/

#include <stdio.h>        // General libraries
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <sys/time.h>
#include <unistd.h>
#include <signal.h>
#include <sys/wait.h>

#define MAX_CONNECTIONS      32     // I believe this is the limit for Quake??
#define BUFFER_LENGTH      2048     // No message should be bigger than this!!!

#include "qipx.h"

char buffer[BUFFER_LENGTH];         // Static buffer for storing messages

char *ProgramName;
char RealProgramName[] = "qipxserver";

char msgQuakeBroadcast[] = { 0x80, 0x00, 0x00, 0x0C,
                             0x02, 'Q' , 'U' , 'A' ,
                             'K' , 'E' , 0x00, 0x03 };
                             
#define lenQuakeBroadcast sizeof(msgQuakeBroadcast)

long IP_ADDRESS_START = 0x7FACED01; // Where to start using IP addresses from

int otherpid;   // Process ID of sibling after it has forked so signal
                // handler can kill them both off if nessesary

int meetingport = 26000;   // In IPX
int serverport = 26000;    // Where Quake server is
int messagelevel = 1;
FILE *messagefile = stdout;

void HandleCommandline(int argc, char *argv[])
{
  int opt; 
  
  ProgramName = argv[0];
  
  while( (opt=getopt(argc,argv,"p:m:s:a:o:h")) != EOF)
  {
    switch(opt)
    {
      case '?':
      case 'h':
        printf("qipxserver v1.0 - allow IPX client to connect to the UDP server on this computer\n"
               "Copyright Oct 1997 by Martijn van Oosterhout. See copyright file for details\n"
               "qipxserver [-p port] [-s port] [-a addr] [-o file] [-m messagelevel] [-h]\n"
               "  -p port          Changes port to react to (default 26000)\n"
               "  -s port          Port the Quake server is on (default 26000)\n"
               "  -a xx.xx.xx.xx   Specifiy starting port for new IP addresses (default 127.172.237.1)\n"
               "  -o file          Specify file to log output to (default stdout)\n"
               "  -m messagelevel  Changes verbosity of messages (default 1)\n"
               "                   0. No messages\n"
               "                   1. Status messages\n"
               "                   2. Displaying of network addresses\n"
               "                   3. Dumping of negotiation packets\n"
               "                   4. Dumping of all packets (_LOTS_ of data!!)\n");
        exit(0);
      case 'p':
        meetingport = atoi(optarg);
        if(meetingport == 0)
        {
          printf("qipxserver: Illegal port number\n");
          exit(1);
        }
        break;
      case 's':
        serverport = atoi(optarg);
        if(serverport == 0)
        {
          printf("qipxserver: Illegal port number\n");
          exit(1);
        }
        break;
      case 'm':
        messagelevel = atoi(optarg);
        break;
      case 'a':
        if((IP_ADDRESS_START = inet_addr(optarg)) == -1)
        {
          fprintf(stderr,"qipxserver: Invalid address\n");
          exit(1);
        }
        if((IP_ADDRESS_START & 0xFF000000) != 0x7F000000)
        {
          fprintf(stderr,"qipxserver: Address must start with 127\n");
          exit(1);
        }
        break;
      case 'o':
        messagefile = fopen(optarg,"wt");
        
        if(!messagefile)
        {
          fprintf(stderr,"qipxserver: Can't open output file '%s' (%s)\n",optarg,strerror(errno));
          exit(0);
        }
        break;
    }
  }
}

void GetServer(int udpsock, struct sockaddr_in *addr)
{
  int length,addrlen,foundone;
  
  struct sockaddr_in destaddr;
  
  addrlen = sizeof(struct sockaddr_ipx);
  destaddr.sin_family = AF_INET;               // Setup send to UDP:serverport
  destaddr.sin_port = htons(serverport);
  destaddr.sin_addr.s_addr = htonl(0x7F000001);
  
  message(2,"qipxserver: Server address is %s\n",PrintUDPaddress(&destaddr));
  
  message(1,"qipxserver: Looking for Quake server\n");
  messagedump(3,msgQuakeBroadcast,lenQuakeBroadcast);
  // Send broadcast to find servers
  sendto(udpsock,msgQuakeBroadcast,lenQuakeBroadcast,0,(struct sockaddr*)&destaddr,sizeof(destaddr));
  
  foundone = 0;
  for(;;)
  {
    int result;
  
    fd_set set;
  
    struct timeval tv;
  
    tv.tv_sec = 2;
    tv.tv_usec = 0;
  
    FD_ZERO(&set);
  
    FD_SET(udpsock,&set);
  
    // Wait for two seconds for a response
    Check(result = select(udpsock+1,&set,NULL,NULL,&tv),"Select");
  
    if(!result)
      break;
     
    if(FD_ISSET(udpsock,&set))
    {
      addrlen = sizeof(struct sockaddr_in);
      Check(length = recvfrom(udpsock,buffer+4,BUFFER_LENGTH-4,0,(struct sockaddr*)addr,&addrlen),"recvfrom");
      
      message(3,"qipxserver: Got reply message\n");
      messagedump(3,buffer,length+4);

      if(length > 10 && buffer[8] == 0x83)
      {
        char *str = &buffer[9];
        foundone = 1;
        
        printf("%s ",str);
        str += strlen(str)+1;

        printf("%s ",str);
        str += strlen(str)+1;

        printf("%s ",str);
        str += strlen(str)+1;
        
        printf("%d/%d\n",str[0],str[1]);
        
        break;
      }
    }
  }
  
  if(foundone == 0)
  {
    message(0,"qipxserver: No server found. Quitting\n");
    exit(0);
  }
}

// To kill off friend if something goes wrong
void KillBuddy(int status)
{
  kill(otherpid,SIGTERM);
  if( status == SIGALRM )
    message(0,"Quitting due to timeout\n");
  else
    message(0,"Quitting due to signal %d\n", status);
  exit(0);
}

// This function handles transfers between IPX:ipxsock and UDP:udpsock
// Destination address for IPX:ipxsock is ipx_addr
// Destination address for UDP:udpsock is udp_addr

void ChildProcess(int meetingsock, int udp_sock, struct sockaddr* ipx_addr)
{
  int length;      // Length of received messages
  int pid;         // Process id the the child
  int newipxsock;  // Handle of new socket
  int localserverport; // Port of IPX socket at this end
  
  struct sockaddr_in serveraddr;   // Destination for UDP socket
  int serveraddrlen;
  
  Check(newipxsock = openipxsocket(0), "Child open");   // Make the new output socket
  
  fclose(stdin);
  
  {
    struct sockaddr_ipx temp;
  
    int tempaddrlen = sizeof(temp);
    Check(getsockname(newipxsock,(struct sockaddr*)&temp,&tempaddrlen),"Getsockname");
    localserverport = htons(temp.sipx_port);
  }

  message(2,"qipxserver: Waiting for reply from Quake server\n");
  
  Check(length = recvfrom(udp_sock, buffer+4, BUFFER_LENGTH-4, 0,(struct sockaddr*)&serveraddr,&serveraddrlen), "Child recv");   // Receive reply from Quake
  *(long*)buffer = 0;
  
  if(length != 9)                                    // Better be the right one!!!
  {
    message(1,"qipxserver: Got bad header (length = %d)\n",length);
    messagedump(3,buffer,length+4);
    exit(0);
  }
  
  serveraddr.sin_port = htons(*(short*)(buffer+9));       // Extract port address
  *(short*)(buffer+9) = localserverport;
  
  message(2,"qipxserver: Got reply from Quake, port = %d\n",htons(serveraddr.sin_port));
  
  // Send the reply from Quake
  Check(sendto(meetingsock,buffer,length+4,0,ipx_addr,sizeof(struct sockaddr_ipx)), "Child sendto");
  
  message(2,"qipxserver: Sent reply to %s\n",PrintIPXaddress((struct sockaddr_ipx*)ipx_addr));
  
  close(meetingsock);
  
  mysignal(SIGCHLD,KillBuddy);
  mysignal(SIGPIPE,KillBuddy);
  mysignal(SIGALRM,KillBuddy);
  
  alarm(60);
  
  message(1,"qipxserver: Starting proxy\n");
  
  Check(pid = fork(), "Child fork")
  
  if(pid)    // Parent. Handles Quake->Client    (UDP->IPX)
  {
    int count = 1;
    
    otherpid = pid;
    SetProgramName("qipxserver: UDP(%s) -> IPX(%s)",PrintUDPaddress(&serveraddr),PrintIPXaddress((struct sockaddr_ipx*)&ipx_addr));
    
    for(;;)
    {
      // Receive from UDP
      Check(length = recv(udp_sock, buffer+4, BUFFER_LENGTH-4, 0), "Child recv");
      *(long*)buffer = count++;
      Check(sendto(newipxsock,buffer,length+4,0,ipx_addr, sizeof(struct sockaddr_ipx)), "Child sendto");
      
      message(4,"qipxserver: Server->Client\n");
      messagedump(4,buffer,length+4);
      
      alarm(240);
    }
  }
  else      // Child. Handles Client->Quake     (IPX->UDP)
  {
    SetProgramName("qipxserver: IPX(%s) -> UDP(%s)",PrintIPXaddress((struct sockaddr_ipx*)&ipx_addr),PrintUDPaddress(&serveraddr));
    otherpid = getppid();
    
    for(;;)
    {
      Check(length = recv(newipxsock,buffer,BUFFER_LENGTH,0), "Child Child recv"); // Receive data
      Check(sendto(udp_sock, buffer+4, length-4, 0, (struct sockaddr*)&serveraddr, sizeof(serveraddr)), "Child Child sendto");
      
      message(4,"qipxserver: Client->Server\n");
      messagedump(4,buffer,length);
      
      if( *(long*)(buffer+4) == htonl( 0x00100009 ) && buffer[12] == 0x02)  // Disconnect packet
      {
        message(1,"qipxserver: Client quitting\n");
        
        kill( otherpid, SIGTERM );
        exit(0);
      }
      
      alarm(240);
    }
  }
}

void ChildDeathHandler(int status)      // Fudge unused parameter
{
  wait(&status); // Clean up child    // Use rubbish parameter
}

void main(int argc, char *argv[])
{
  int meetingsock;                 // Handle of the meeting socket
  int infosock;                    // Socket for extracting info from server
  int length;                      // Length of the read data
  struct sockaddr_ipx clientaddr;  // For the client address
  int clientaddrlen;               // The length of clientaddr
  struct sockaddr_in serveraddr;   // Destination address for server
  int pid;                         // PID of child process
  int numconnections=0;            // Number of connections so far
  
  HandleCommandline(argc,argv);
  
  infosock = openudpsocket(0);
  
  GetServer(infosock,&serveraddr);
  
  meetingsock = openipxsocket(meetingport);         // Open the socket IPX:26000
  
  mysignal(SIGCHLD,ChildDeathHandler);
  
  for(;;)
  {
    // Wait for a packet
    message(1,"qipxserver Parent: Waiting for client...\n");
    clientaddrlen = sizeof(clientaddr);
    Check(length = recvfrom(meetingsock,buffer,BUFFER_LENGTH,0,(struct sockaddr*)&clientaddr,&clientaddrlen), "recvfrom");
  
    // Print source
    message(1,"qipxserver Parent: Received from %s\n",PrintIPXaddress(&clientaddr));
    
    if(length != 16 || (buffer[8] != 0x01 && buffer[8] != 0x02))
    {
      message(1,"qipxserver Parent: Not a connections request, type %02X\n",buffer[8]);
      messagedump(3,buffer,length);
      
      Check(sendto(infosock,buffer+4,length-4,0,(struct sockaddr*)&serveraddr,sizeof(serveraddr)), "Sendto");
      message(2,"qipxserver Parent: Forwarded message to Quake server\n");
      
      if( !fork() )
      {
        alarm(10);  // Wait up to 10secs for a reply
      
        Check(length = recv(infosock,buffer+4,BUFFER_LENGTH-4,0),"Recv");
      
        message(3,"qipxserver Parent: Server reply\n");
        messagedump(3,buffer,length+4);
      
        Check( sendto( meetingsock, buffer, length+4, 0, (struct sockaddr*)&clientaddr, clientaddrlen), "sendto" );
        
        exit(0);
      }
    }
    
    if(buffer[8] == 0x01)
    {
      // Make a new connection
      int udp_sock = openudpsocket2(0,IP_ADDRESS_START+numconnections);
      
      message(1,"qipxserver Parent: New person, connnection number %d\n",numconnections);
      
      Check(pid = fork(),"Fork");
      
      if(!pid)  // If child
      {
        ChildProcess(meetingsock,udp_sock,(struct sockaddr*)&clientaddr);     // Should never return
        exit(0);
      }
      
      Check(sendto(udp_sock,buffer+4,length-4,0,(struct sockaddr*)&serveraddr,sizeof(serveraddr)), "Sendto");
      message(2,"qipxserver Parent: Forwarded message to Quake server\n");
      numconnections++;
    
      close(udp_sock);
    }
    else if(buffer[8] == 0x02)
    {
      message(1,"qipxserver Parent: Server broadcast\n");
      Check(sendto(infosock,buffer+4,length-4,0,(struct sockaddr*)&serveraddr,sizeof(serveraddr)), "Sendto");
      Check(length = recv(infosock,buffer+4,BUFFER_LENGTH-4,0),"Recv");
      
      message(3,"qipxserver Parent: Server broadcast reply\n");
      messagedump(3,buffer,length+4);
      
      if(length < 10 || buffer[8] != 0x83)
      {
        message(1,"qipxserver Parent: Not expected reply\n");
        continue;
      }
      
      {
        struct sockaddr_ipx temp;
        int templength = sizeof(temp);
        char *addr, *rest = &buffer[9] + strlen(&buffer[9]) + 1;
        
        Check(getsockname(meetingsock,(struct sockaddr*)&temp,&templength),"Getsockname");
        addr = PrintIPXaddress(&temp);
        
        memmove(&buffer[9] + strlen(addr) + 1,rest,length);
        strcpy(&buffer[9],addr);
  
        length = length - (rest-buffer) + 14 + strlen(addr);
        *(short*)(buffer+6) = htons(length-4); // Fixup length
        messagedump(3,buffer,length);
        
        Check(sendto(meetingsock,buffer,length,0,(struct sockaddr*)&clientaddr,clientaddrlen),"Sendto");
      }
    }
  }
}

