/********************************************
 Plan for quakelog
 
 On startup
 1. Broadcast for UDP servers
 2. Give user choice of server
 
 For the proxy
 3. Open an UDP:27000 socket
 4. Read from socket (will block)
 5. Make sure return address is ourselves
 6. Check that it is a signon message
 7. Forward the signon message
 8. In reply message, fudge the socket number to our socket
 9. Fork and handle messages in both directions

*******************************************/

#include <stdio.h>        // General libraries
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <sys/time.h>
#include <unistd.h>
#include <signal.h>

#include <sys/socket.h>   // General socket stuff
#include <netdb.h>

#include <arpa/inet.h>    // TCP/IP   (UDP)
#include <linux/ipx.h>    // IPX

#include "qipx.h"

#define MAX_CONNECTIONS      32     // Maximum expected servers on the current network
#define BUFFER_LENGTH      2048     // No message should be bigger than this!!!

char buffer[BUFFER_LENGTH];         // Static buffer for storing messages

char *ProgramName;
char RealProgramName[] = "quakelog";

char msgQuakeBroadcast[] = { 0x80, 0x00, 0x00, 0x0C,
                             0x02, 'Q' , 'U' , 'A' ,
                             'K' , 'E' , 0x00, 0x03 };
                             
#define lenQuakeBroadcast sizeof(msgQuakeBroadcast)

FILE *messagefile = stdout;

int portnum = 27000;  // Port number to use
int serverport = 26000;
int messagelevel = 1; // messagelevel

int otherpid;

void HandleCommandline(int argc, char *argv[])
{
  int opt; 
  
  while( (opt=getopt(argc,argv,"p:m:s:h")) != EOF)
  {
    switch(opt)
    {
      case '?':
      case 'h':
        printf("quakelog v1.0 - log the messages of a network quake game\n"
               "Copyright Oct 1997 by Martijn van Oosterhout. See copyright file for details\n"              
               "quakelog [-p port] [-s port] [-m messagelevel] [-h]\n"
               "  -p port          Changes port to react to (default 27000)\n"
               "  -s port          Port the Quake server is on (default 26000)\n"
               "  -m messagelevel  Changes verbosity of messages (default 1)\n"
               "                   0. No messages\n"
               "                   1. Status messages\n"
               "                   2. Displaying of network addresses\n"
               "                   3. Dumping of negotiation packets\n"
               "                   4. Dumping of all packet (_LOTS_ of data!!)\n");
        exit(0);
      case 'p':
        portnum = atoi(optarg);
        if(portnum == 0)
        {
          printf("Illegal port number\n");
          exit(1);
        }
        break;
      case 's':
        serverport = atoi(optarg);
        if(serverport == 0)
        {
          printf("Illegal port number\n");
          exit(1);
        }
        break;
      case 'm':
        messagelevel = atoi(optarg);
        break;
    }
  }
}

void GetServer(int udpsock, struct sockaddr_in *addr)
{
  struct sockaddr_in sources[MAX_CONNECTIONS];
  int numsources = 0;
  int length,addrlen,choice;
  
  struct sockaddr_in destaddr;
  
  addrlen = sizeof(struct sockaddr_in);
  destaddr.sin_family = AF_INET;
  destaddr.sin_port = htons(serverport);
  destaddr.sin_addr.s_addr = INADDR_ANY;
  
  message(2,"Broadcast address is %s\n",PrintUDPaddress(&destaddr));
  
  message(1,"Broadcasting for Quake servers\n");
  messagedump(3,msgQuakeBroadcast,lenQuakeBroadcast);
  // Send broadcast to find servers
  sendto(udpsock,msgQuakeBroadcast,lenQuakeBroadcast,0,(struct sockaddr*)&destaddr,sizeof(destaddr));
  
  while(numsources < MAX_CONNECTIONS)
  {
    int result;
  
    fd_set set;
  
    struct timeval tv;
  
    tv.tv_sec = 2;
    tv.tv_usec = 0;
  
    FD_ZERO(&set);
  
    FD_SET(udpsock,&set);
  
    // Wait for two seconds for a response
    Check(result = select(udpsock+1,&set,NULL,NULL,&tv),"Select");
  
    if(!result)
      break;
     
    if(FD_ISSET(udpsock,&set))
    {
      addrlen = sizeof(struct sockaddr_in);
      Check(length = recvfrom(udpsock,buffer,BUFFER_LENGTH,0,(struct sockaddr*)&sources[numsources],&addrlen),"recvfrom");
      
      message(3,"Got reply message\n");
      messagedump(3,buffer,length);

      if(length > 10 && buffer[4] == 0x83)
      {
        char *str = &buffer[5];
        
        printf("%d. %s ",numsources+1,str);
        str += strlen(str)+1;

        printf("%s ",str);
        str += strlen(str)+1;

        printf("%s ",str);
        str += strlen(str)+1;
        
        printf("%d/%d\n",str[0],str[1]);
        
        numsources++;
      }
    }
  }
  
  if(numsources == 0)
  {
    printf("No servers found. Quitting\n");
    exit(0);
  }

  if(numsources !=  1)
  {  
    printf("\nMake a choice (blank to quit): ");
    fflush(stdout);
  
    fgets(buffer,10,stdin);
  
    choice = atoi(buffer);
  
    if( choice < 1 || choice > numsources)
    {
      printf("Quitting\n");
      exit(0);
    }
  }
  else
  {
    printf("Automatically choosing only one\n");
    choice = 1;
  }
  
  memcpy(addr,&sources[choice-1],sizeof(struct sockaddr_in));
  
  message(1,"Chose server %s\n",PrintUDPaddress(addr));
}

void main(int argc, char *argv[])
{
  int udp27000sock;            // Handle of socket UDP:27000
  int udpsock;                 // Handle of other UDP socket
  int udpserversock;           // Handle of another socket
  
  int length;                  // Length of the read data
  
  struct sockaddr_in addr;     // For receiving the address of the client
  int addrlen;                 // The length of the IPX in readfrom
  
  struct sockaddr_in destaddr; // Broadcast address UDP:26000
  int destaddrlen;             // Length of that
  
  int localserversocket;       // Local udpsocket
  
  int pid;
  
  ProgramName = argv[0];
  
  HandleCommandline(argc,argv);
  
  udpserversock = openudpsocket(0);         // Open the server socket
  
  message(1,"Checking for servers\n");
  
  GetServer(udpserversock,&destaddr);  // Get replies and ask user
  
  // destaddr now contains address of server to send connect to.
  
  udp27000sock = openudpsocket(portnum);  // Open server socket
  udpsock = openudpsocket(0);       // Open communication socket
  
  addrlen = sizeof(addr);
  Check(getsockname(udpsock,(struct sockaddr*)&addr,&addrlen),"Getsockname");
  localserversocket = htons(addr.sin_port);
  
  message(2,"Local server socket is %d\n",localserversocket);

  for(;;) // Until connection is made
  {
    for(;;)
    {
      message(1,"Waiting for client...\n");
      
      // Now wait for client to join.
      addrlen = sizeof(addr);
      Check(length = recvfrom(udp27000sock,buffer,BUFFER_LENGTH,0,(struct sockaddr*)&addr,&addrlen), "recvfrom");   // Wait for a packet
      // addr now contains address of client
    
      if(length != 12 || buffer[4] != 0x01)
      {
        message(1,"quakelog: Not a connect message\n");
        messagedump(3,buffer,length);
        continue;
      }
      
      message(1,"Received connect message\n");
      messagedump(3,buffer,length);
      break;
    }
    
    message(2,"Client address : %s\n",PrintUDPaddress(&addr));
    message(2,"Server address : %s\n",PrintUDPaddress(&destaddr));
    
    // Forward connect message
    destaddrlen = sizeof(destaddr);
    Check(sendto(udpserversock,buffer,length,0,(struct sockaddr*)&destaddr,destaddrlen),"Sendto"); 
    
    message(1,"Waiting for server reply\n");
  
    // Receive server message
    Check(length = recv(udpserversock,buffer,BUFFER_LENGTH,0),"recv");
  
    if(buffer[4] == 0x81)
    {
      message(1,"Connection accepted\n");
      messagedump(3,buffer,length);
      
      // Connection accepted
      // Set destaddr to point to server communication address
      destaddr.sin_port = htons(*(unsigned short*)(buffer+5));
      
      message(2,"New server address : %s\n",PrintUDPaddress(&destaddr));
      
      // Tell client to use the local server port
      *(short*)(buffer+5) = localserversocket;
    
      message(3,"Forwarding modified message\n");
      messagedump(3,buffer,length);
      
      // Forward message to client
      Check(sendto(udp27000sock,buffer,length,0,(struct sockaddr*)&addr,sizeof(addr)),"Sendto");
    
      // Negotiation complete, start proxy
      break;
    }
  
    message(1,"Connection refused\n");
    messagedump(3,buffer,length);
    // Connection refused. Forward message unmodified
    Check(sendto(udp27000sock,buffer,length,0,(struct sockaddr*)&addr,sizeof(addr)),"Sendto");
  }
  
  // Starting proxy
  message(1,"Starting proxy\n");
  
  close(udp27000sock); // Close negoiation port. Run multiple copies??
  
  Check(pid = fork(), "Fork");
  
  if(pid) // Parent, server->client
  {
    otherpid = pid;
    
    for(;;)
    {
      Check(length = recv(udpserversock,buffer,BUFFER_LENGTH,0),"Parent recv");  
      Check(sendto(udpsock,buffer,length,0,(struct sockaddr*)&addr,sizeof(addr)),"Parent sendto");
      message(4,"Server->Client\n");
      messagedump(4,buffer,length);
    }
  }
  else // Child, client->server
  {
    otherpid = getppid();
    
    for(;;)
    {
      Check(length = recv(udpsock,buffer,BUFFER_LENGTH,0),"Child recv");
      Check(sendto(udpserversock,buffer,length,0,(struct sockaddr*)&destaddr,sizeof(destaddr)),"Child sendto");
      message(4,"Client->Server\n");
      messagedump(4,buffer,length);
    }
  }
}
